package webhook

import (
	"testing"

	"code.justin.tv/dta/skadi/api"
	"github.com/google/go-github/github"
)

func commitStatus(state, branch string) *CommitStatusEvent {
	return &CommitStatusEvent{
		State: &state,
		Branches: []*github.Branch{
			{
				Name: &branch,
			},
		},
	}
}

func buildConfig(branch string, p, s, f bool) *api.ChatBranchesConfig {
	return &api.ChatBranchesConfig{
		Branches: map[string]api.ChatSectionConfig{
			branch: {
				Success: s,
				Pending: p,
				Failure: f,
			},
		},
	}
}

func TestShouldBroadcastCommitStatus(t *testing.T) {
	for i, set := range []struct {
		event    *CommitStatusEvent
		config   *api.ChatBranchesConfig
		expected bool
	}{
		{commitStatus("pending", "master"), &api.ChatBranchesConfig{}, false},
		{commitStatus("success", "master"), buildConfig("master", false, true, false), true},
		{commitStatus("failure", "foo"), buildConfig("*", false, false, true), true},
	} {
		if shouldBroadcastCommitStatus(set.event, set.config) != set.expected {
			t.Fatalf("unexpected status broadcast. Set Index: %v", i)
		}
	}
}

func TestFormatCommitMessage(t *testing.T) {
	commit := `> did "things" & 'stuff' <`
	url := "http://www.zombo.com/"
	expected := `<http://www.zombo.com/|&gt; did "things" &amp; 'stuff' &lt;>`

	text := formatCommitMessage(url, commit)

	if text != expected {
		t.Fatalf("commit message not formatted as expected for chat broadcast:\n%v", text)
	}
}

func TestFormatPullRequest(t *testing.T) {
	commit := `Merge pull request #1 from dta/skadi

did stuff`
	url := "http://www.zombo.com/"
	expected := `<http://www.zombo.com/|Merge pull request #1 from dta/skadi>

did stuff`

	text := formatCommitMessage(url, commit)

	if text != expected {
		t.Fatalf("commit message not formatted as expected for chat broadcast:\n%v", text)
	}
}
