package main

import (
	"code.justin.tv/dta/skadi/pkg/email"
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"net/http/pprof"
	"net/url"
	"os"
	"path"
	"strconv"
	"strings"

	"time"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/golibs/statsd"
	"code.justin.tv/dta/skadi/internal/webhook"
	"code.justin.tv/dta/skadi/pkg/appconfig"
	"code.justin.tv/dta/skadi/pkg/artifact"
	"code.justin.tv/dta/skadi/pkg/build"
	"code.justin.tv/dta/skadi/pkg/candidate"
	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/deploystatus"
	"code.justin.tv/dta/skadi/pkg/environment"
	"code.justin.tv/dta/skadi/pkg/freeze"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"code.justin.tv/dta/skadi/pkg/helpers"
	"code.justin.tv/dta/skadi/pkg/info"
	"code.justin.tv/dta/skadi/pkg/infoapi"
	"code.justin.tv/dta/skadi/pkg/motd"
	skadiOauth "code.justin.tv/dta/skadi/pkg/oauth"
	"code.justin.tv/dta/skadi/pkg/repo"
	"code.justin.tv/dta/skadi/pkg/rtevent_receiver"
	"code.justin.tv/dta/skadi/pkg/subscriptions"
	"code.justin.tv/dta/skadi/pkg/user"
	"code.justin.tv/release/jenkins-api"
	"code.justin.tv/release/twitch-consul-api"
	log "github.com/Sirupsen/logrus"
	cactusstatsd "github.com/cactus/go-statsd-client/statsd"
	"github.com/google/go-github/github"
	"github.com/gorilla/handlers"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/jmoiron/sqlx"
	_ "github.com/lib/pq"
	"github.com/nlopes/slack"
	"github.com/stvp/rollbar"
	"golang.org/x/net/context"
	"golang.org/x/oauth2"
)

var (
	slackClient        *slack.Client
	jenkinsClient      *jenkins.Client
	githubClient       *github.Client
	consulClient       *consulapi.Client
	deployStatsdClient statsd.Stats
	metricStatsdClient cactusstatsd.Statter
	// GitCommit is set at build time to the current git commit for the project.
	GitCommit   = "development"
	oauthConfig = &oauth2.Config{
		Scopes: []string{skadiOauth.OauthScope},
	}
)

var c *appconfig.AppConfig

func init() {
	infoGlobal := info.NewInfo("global")
	infoGlobal.SetProperty("StartedAt", time.Now().String())
	infoGlobal.SetProperty("CmdLine", strings.Join(os.Args, " "))
	log.SetFormatter(&log.JSONFormatter{})

	fmt.Fprintf(os.Stderr, "Running commit %q with config:\n", GitCommit)

	// Get config
	c = appconfig.Get()

	loglevel, llerr := log.ParseLevel(c.LogLevel)
	if llerr != nil {
		log.Fatalf("Unable to set log level (%s): %s", c.LogLevel, llerr)
	}
	log.SetLevel(loglevel)

	slackClient = slack.New(c.SlackAPIToken)

	var err error
	jenkinsClient = jenkins.NewClient(c.JenkinsHost, chitin.Client(context.Background()))
	jenkinsClient.SetAuth(c.JenkinsUser, c.JenkinsAPIToken)

	githubClient, err = helpers.ConfigureGithubClient(
		c.GithubHost,
		c.GithubAPIToken,
		context.Background(),
		c.InsecureGithub,
	)
	if err != nil {
		log.Fatal(err)
	}

	consulClient, err = twitchconsulapi.NewClient(c.ConsulHost, chitin.Client(context.Background()))
	if err != nil {
		log.Fatalf("error creating consulapi client: %v", err)
	}

	statsdAddr := net.JoinHostPort(c.StatsdHost, strconv.Itoa(c.StatsdPort))
	deployStatsdClient, err = statsd.Dial("udp", statsdAddr, statsd.StatsConfig{
		Rate:   1.0,
		Prefix: c.DeployStatsdPrefix,
	})
	if err != nil {
		log.Fatalf("error creating statsd client: %v", err)
	}

	// Consul is liberal in what it accepts as a valid key but specific in what
	// it returns. By cleaning and removing any leading '/' we can make our
	// paths match what it returns.
	c.ConsulPrefix = strings.TrimLeft(path.Clean(c.ConsulPrefix), "/")

	oauthConfig.ClientID = c.GithubClientID
	oauthConfig.ClientSecret = c.GithubClientSecret
	oauthConfig.Endpoint = oauth2.Endpoint{
		AuthURL:  c.GithubAuthURL,
		TokenURL: c.GithubTokenURL,
	}
	oauth2.RegisterBrokenAuthHeaderProvider(c.GithubTokenURL)
	skadiOauth.InsecureGithub = c.InsecureGithub

	if c.MetricStatsdPrefix == "" {
		c.MetricStatsdPrefix = strings.Join([]string{"skadi", c.Environment}, ".")
	}

	metricStatsdClient, err = cactusstatsd.NewBufferedClient(statsdAddr, c.MetricStatsdPrefix, time.Second, 0)
	if err != nil {
		log.Fatalf("error creating statsd client: %v", err)
	}

	config.StatsdClient = metricStatsdClient
	config.JenkinsDefaultClient = jenkinsClient
	config.GithubDefaultToken = c.GithubAPIToken
	config.GithubHost = c.GithubHost
	config.InsecureGithub = c.InsecureGithub
	webhook.BaseURL, err = url.Parse(c.APIBaseURL)
	if err != nil {
		log.Fatalf("Can't parse api-base-url parameter: %v", err)
	}

	// Configure Rollbar
	rollbar.Token = c.RollbarToken
	rollbar.Environment = c.Environment
	rollbar.CodeVersion = GitCommit

	// Configure the url for the webhook we register:
	repo.WebhookURL = c.WebhookURL
	repo.InsecureGithub = c.InsecureGithub
	repo.InstallGitWebhook = c.InstallGitWebhook

	// Run periodic config loader
	appconfig.RunConfigUpdater(consulClient)
}

func healthCheck(w http.ResponseWriter, req *http.Request) {
	type hcStruct struct {
		GitCommit string
	}

	if err := json.NewEncoder(w).Encode(&hcStruct{
		GitCommit: GitCommit,
	}); err != nil {
		log.Printf("Error encoding healthCheck: %v", err)
	}
}

func main() {
	defer func() {
		if r := recover(); r != nil {
			rollbar.Error(rollbar.ERR, fmt.Errorf("uncaught panic: %v", r))
		}
	}()

	workerID, err := helpers.GetLocalIP()
	if err != nil {
		log.Fatalf("Failed to get local IP address. - %v", err)
	}
	log.Printf("WorkerID=%v", workerID)
	db, err := sqlx.Open("postgres", c.PgConnInfo)
	if err != nil {
		log.Fatal(err)
	}
	defer db.Close()
	if err := db.Ping(); err != nil {
		log.Fatalf("Can't connect to db: %v", err)
	}

	router := mux.NewRouter()
	config.CreateHandler(router, "/running", healthCheck, nil)

	wc := &webhook.WebhookConfig{
		ConsulPrefix:  c.ConsulPrefix,
		SlackClient:   slackClient,
		JenkinsClient: jenkinsClient,
		GithubClient:  githubClient,
		ConsulClient:  consulClient,
		StatsdClient:  deployStatsdClient,
		DqConfig: &webhook.DeployQueueConfig{
			QueueURL:              c.QueueURL,
			QueueRegion:           c.QueueRegion,
			EnableQueueLog:        c.EnableQueueLog,
			QueueLogPath:          c.QueueLogPath,
			QueueRetryIntervalMin: (time.Duration(c.QueueRetryIntervalMin) * time.Millisecond),
			QueueRetryIntervalMax: (time.Duration(c.QueueRetryIntervalMin) * time.Millisecond),
			QueueRetryRuntime:     (time.Duration(c.QueueRetryIntervalMin) * time.Millisecond),
		},
		DfConfig: &webhook.DeployFactoryConfig{
			MaxConcurrentDeployers: c.MaxConcurrentDeployers,
			ConsiderStallSecs:      c.ConsiderStallSecs,
		},
	}
	if err := wc.Start(); err != nil {
		log.Fatal(err)
	}

	fc := &freeze.FreezeConfig{
		GithubClient: githubClient,
		ConsulClient: consulClient,
		SlackClient:  slackClient,
		BaseURL:      webhook.BaseURL,
	}

	email.Init(c.SESRegion, c.EmailSender, "UTF-8")
	githubcache.Init(githubClient)
	registerUIHandlers(router)
	wc.RegisterHandlers(router)
	artifact.RegisterHandlers(router, db, consulClient)
	build.RegisterHandlers(router)
	candidate.RegisterHandlers(router, consulClient)
	deployment.SetRuntimeEnvironmentInfo(c.Environment, c.Environment == config.RuntimeEnvironmentNameProduction)
	deployment.RegisterHandlers(router, workerID, db, consulClient)
	environment.RegisterHandlers(router, db)
	repo.AddNoHookCheckUsers(c.InstallGitWebhookSkipUsers)
	repo.RegisterOldHandlers(router)
	repo.RegisterHandlers(router, consulClient)
	user.RegisterHandlers(router)
	subscriptions.RegisterHandlers(router, db)
	freeze.RegisterHandlers(router, db, fc)
	skadiOauth.RegisterHandlers(router, oauthConfig)
	motd.RegisterHandlers(router, githubClient, c.MotdRepo)
	infoapi.RegisterHandlers(router)
	deploystatus.RegisterHandlers(router, db, consulClient, &c.ConsulDatacenters)
	deployment.AutoDeployRunner(consulClient, githubClient)

	if c.EnableRteventReceiver {
		rtevent_receiver.LaunchRteventReceiverThread(db, c.RteventLogPath)
	}

	if c.EnablePprof {
		registerPprofHandlers(router)
	}

	bind := net.JoinHostPort("", strconv.Itoa(c.Port))
	log.Printf("Listening on %q", bind)
	log.Fatal(http.ListenAndServe(
		bind,
		superHandler(handlers.CombinedLoggingHandler(os.Stdout,
			chitin.Handler(
				router,
				chitin.SetBaseContext(mustTraceContext()),
			),
		)),
	))
}

func registerPprofHandlers(r *mux.Router) {
	config.CreateHandler(r, "/debug/pprof", func(w http.ResponseWriter, r *http.Request) {
		http.Redirect(w, r, "/debug/pprof/", http.StatusMovedPermanently)
	}, nil)
	config.CreateHandler(r, "/debug/pprof/", pprof.Index, nil)
	config.CreateHandler(r, "/debug/pprof/cmdline", pprof.Cmdline, nil)
	config.CreateHandler(r, "/debug/pprof/profile", pprof.Profile, nil)
	config.CreateHandler(r, "/debug/pprof/symbol", pprof.Symbol, nil)
	config.CreateHandler(r, "/debug/pprof/trace", pprof.Trace, nil)
	config.CreateHandler(r, "/debug/pprof/{*}", pprof.Index, nil)
}

// superHandler is used as a preprocessor that wraps given handler.
// Anything that needs to be handled commonly before or after actual handler
// logic can be defined here.
func superHandler(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// set user's real address when forwarded by ELB
		addr := r.Header.Get("X-Forwarded-For")
		if len(addr) > 0 {
			r.RemoteAddr = addr
		}
		// check protocol
		if c.HttpsOnly {
			if r.Header.Get("X-Forwarded-Proto") == "http" && strings.HasPrefix(c.APIBaseURL, "https://") {
				newUrl := c.APIBaseURL + r.URL.Path
				if len(r.URL.RawQuery) > 0 {
					newUrl += "?" + r.URL.RawQuery
				}
				http.Redirect(w, r, newUrl, http.StatusPermanentRedirect)
				return
			}
		}

		h.ServeHTTP(w, r)
	})
}

func mustTraceContext() context.Context {
	ctx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		log.Fatal(err)
	}

	return ctx
}
