package appconfig

import (
	"fmt"
	"reflect"
	"regexp"
	"strconv"
	"strings"
	"sync"
	"time"

	"code.justin.tv/dta/skadi/pkg/helpers"
	"code.justin.tv/dta/skadi/pkg/info"
	"code.justin.tv/release/libforerunner"
	log "github.com/Sirupsen/logrus"
	consulapi "github.com/hashicorp/consul/api"
)

// "appconfig" package provides global access to the application configuration.
// It is a singleton implementation and the returned AppConfig struct always
// keeps the current and the latest configurations. It checks and reloads
// defined configuration entires, not all of them, from the consul.
//
// The returned AppConfig struct can be accessed same as normal data struct.
// Or you could carry the pointer of specific entries and use it to get the
// latest value. You can also register callback function when some initialization
// is required upon the value change.

const (
	CONFIG_CHECK_INTERVAL_SEC = 60
	CONSUL_CONFIG_KEYS        = "consul-master-datacenters,consul-datacenters"
	CONSUL_CONFIG_PREFIX      = "settings/skadi"
)

var (
	g_conf      *AppConfig
	callbackFns []CallbackType
	once        sync.Once

	infoConfig = info.NewInfo("config")
	re_snake   = regexp.MustCompile("(^[A-Za-z])|-([A-Za-z])")
)

func init() {
	c := &AppConfig{
		EnablePprof:             true,
		ConsulHost:              "api.us-west-2.prod.consul.live-video.a2z.com",
		ConsulPrefix:            "application-data/skadi/dev",
		ConsulMasterDatacenters: "devtools,us-west2",
		ConsulDatacenters:       "devtools,us-west2",
		Port:                    8080,
		PgConnInfo:              "sslmode=disable",
		GithubClientID:          "aaaaaaaaaaaaaaaaaaaa",
		GithubClientSecret:      "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
		GithubHost:              "https://git.xarth.tv/api/v3/",
		GithubAuthURL:           "https://git.xarth.tv/login/oauth/authorize",
		GithubTokenURL:          "https://git.xarth.tv/login/oauth/access_token",
		StatsdHost:              "statsd.central.twitch.a2z.com",
		StatsdPort:              8125,
		RollbarToken:            "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
		DeployStatsdPrefix:      "deploys-development",
		Environment:             "development",
		APIBaseURL:              "http://localhost:8080/",
		InsecureGithub:          false, // Not needed, but I prefer to be verbose about the fact that we require a valid cert.
		WebhookURL:              "https://deploy.xarth.tv/v1/webhook",
		LogLevel:                "warn",
		// DeployFactory related default configuration.
		MaxConcurrentDeployers:     30,
		QueueURL:                   "https://sqs.us-west-2.amazonaws.com/043714768218/skadi_dq_test",
		QueueRegion:                "us-west-2",
		EnableQueueLog:             false,
		QueueLogPath:               "/tmp/skadi_queue.log",
		QueueRetryIntervalMin:      100,
		QueueRetryIntervalMax:      500,
		QueueRetryRuntime:          3000,
		ConsiderStallSecs:          3600,
		EnableRteventReceiver:      false,
		RteventLogPath:             "/tmp/rtevent_queue.log",
		InstallGitWebhook:          true,
		InstallGitWebhookSkipUsers: "devtools",
		MotdRepo:                   "dta/skadi-motd",
		HttpsOnly:                  false,
		SESRegion:                  "us-west-2",
		EmailSender:                "noreply@deploy.twitch.a2z.com",
	}

	fr, err := libforerunner.Init(&libforerunner.Options{
		DefaultConfig: c,
	})
	if err != nil {
		log.Fatal(err)
	}
	err = fr.GetConfig(c)
	if err != nil {
		log.Fatal(err)
	}
	fr.Print()
	g_conf = c
	updateInfoConfig()
	resetCallback()
}

func RunConfigUpdater(cClient *consulapi.Client) {
	once.Do(func() {
		log.Println("Starting appconfig updater")
		go func() {
			entries := helpers.UniqueStringsSorted(helpers.SplitStringNoEmpty(CONSUL_CONFIG_KEYS, ","))
			for {
				time.Sleep(CONFIG_CHECK_INTERVAL_SEC * time.Second)

				found_updates := false
				for _, entry := range entries {
					log.Debugf("Checking appconfig: %s", entry)
					newvalue, err := readConfigFromConsul(cClient, entry)
					if err != nil {
						log.Warnf("failed to retrieve appconfig %s: %v", entry, err)
					}

					// Find matching entry from AppConfig
					v := findEntryFromAppConfig(toCamelCase(entry))
					if v == nil {
						log.Errorf("No matching entry found for %s", entry)
						continue
					}

					// Check if changed
					if v.String() != newvalue {
						switch v.Kind() {
						case reflect.String:
							v.SetString(newvalue)
							found_updates = true
						case reflect.Int:
							newint, err := strconv.ParseInt(newvalue, 10, 64)
							if err == nil {
								v.SetInt(newint)
								found_updates = true
							} else {
								log.Warnf("Invalid data format: not int type - %s", newvalue)
							}
						case reflect.Bool:
							newbool, err := strconv.ParseBool(newvalue)
							if err == nil {
								v.SetBool(newbool)
								found_updates = true
							} else {
								log.Warnf("Invalid data format: not bool type - %s", newvalue)
							}
						default:
							// This should never happen unless we introduce new type in AppConfig
							log.Warnf("Not supported kind: %s", entry)
						}
					}
				}

				if found_updates {
					notifyCallbacks()
				}
			}
		}()
	})
}

func Get() *AppConfig {
	return g_conf
}

func RegisterCallback(fn CallbackType) {
	callbackFns = append(callbackFns, fn)
	// call the callback immediately with current config
	fn(g_conf)
}

func resetCallback() {
	callbackFns = []CallbackType{}
}

func readConfigFromConsul(cClient *consulapi.Client, keyname string) (string, error) {
	keypath := fmt.Sprintf("%v/%v/%v", CONSUL_CONFIG_PREFIX, g_conf.Environment, keyname)
	kv, _, err := cClient.KV().Get(keypath, nil)
	if err != nil {
		return "", err
	}
	if kv == nil {
		return "", fmt.Errorf("No such key: %s", keypath)
	}
	return string(kv.Value), nil
}

func findEntryFromAppConfig(name string) *reflect.Value {
	v := reflect.ValueOf(g_conf).Elem()
	for n := 0; n < v.NumField(); n++ {
		fname := v.Type().Field(n).Name
		if fname == name {
			fval := v.Field(n)
			return &fval
		}
	}
	return nil
}

func notifyCallbacks() {
	for i, fn := range callbackFns {
		log.Debugf("notifying callback #%d", i)
		fn(g_conf)
	}
	updateInfoConfig()
}

// Update expvar info/config
func updateInfoConfig() {
	infoConfig.Init()
	infoConfig.SetStruct(g_conf)
}

func toCamelCase(str string) string {
	return re_snake.ReplaceAllStringFunc(str, func(s string) string {
		return strings.ToUpper(strings.Replace(s, "-", "", -1))
	})
}
