package artifact

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"path"

	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/repo"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/jmoiron/sqlx"
)

var (
	db           *deployment.DB
	consulClient *consulapi.Client
)

// RegisterHandlers will configure all routes for this submodule.
func RegisterHandlers(r *mux.Router, dbInstance *sqlx.DB, cClient *consulapi.Client) {
	consulClient = cClient
	db = &deployment.DB{DB: dbInstance}
	config.CreateHandler(r, "/v1/artifacts/deployed",
		DeployedHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
	config.CreateHandler(r, "/v1/artifacts/download", DownloadHandler, nil).Methods("GET")
	config.CreateHandler(r, "/v1/artifact/deployed", DeployedEnvHandler, &config.RouteOptions{AddCORS: true}).Methods("GET")
	config.CreateHandler(r, "/v1/artifact/compare", CompareDeployedHandler, &config.RouteOptions{AddCORS: true}).Methods("GET")
}

func DeployedHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	rep := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}

	githubClient, err := config.GithubClient(ctx, true)
	if err != nil {
		config.JSONError(w, 500, "error creating github client", err)
		return
	}

	artifacts, err := LoadDeployedArtifacts(consulClient, githubClient, rep)
	if err != nil {
		config.JSONError(w, 500, "error loading deployed artifacts", err)
		return
	}
	json.NewEncoder(w).Encode(artifacts)
}

func DeployedEnvHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	rep := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}
	env := r.FormValue("env")

	githubClient, err := config.GithubClient(ctx, true)
	if err != nil {
		config.JSONError(w, 500, "error creating github client", err)
		return
	}

	artifact, err := LoadDeployedArtifact(consulClient, githubClient, rep, env)
	if err != nil {
		code := 500
		if isEnvNotFoundError(err) {
			code = 409
		}
		config.JSONError(w, code, "error loading deployed artifact", err)
		return
	}

	json.NewEncoder(w).Encode(artifact)
}

func CompareDeployedHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	rep := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}
	env := r.FormValue("env")
	ref := r.FormValue("ref")

	githubClient, err := config.GithubClient(ctx, true)
	if err != nil {
		config.JSONError(w, 500, "error creating github client", err)
		return
	}

	artifact, err := LoadDeployedArtifact(consulClient, githubClient, rep, env)
	if err != nil {
		code := 500
		if isEnvNotFoundError(err) {
			code = 409
		}
		config.JSONError(w, code, "error loading deployed artifact", err)
		return
	}

	previousSHA := artifact.GithubCommit.SHA

	commitsComparison, _, err := githubClient.Repositories.CompareCommits(ctx, rep.Owner, rep.Name, *previousSHA, ref)

	comparison := &CompareDeployed{
		GithubCompare: commitsComparison,
		Ref:           ref,
		PreviousSHA:   *previousSHA,
	}

	json.NewEncoder(w).Encode(comparison)
}

func DownloadHandler(w http.ResponseWriter, r *http.Request) {
	env := r.FormValue("environment")
	rep := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}
	sha := r.FormValue("sha")

	var artifacts map[string]*Artifact
	var artifact *Artifact
	if sha != "" {
		artifact = &Artifact{Repository: rep, SHA: sha, DownloadURL: fmt.Sprintf("s3://%v/%v/%v/%v.tgz", DeployBucket, rep.Owner, rep.Name, sha)}
	} else {
		var err error
		artifacts, err = readDeployedEnvironments(consulClient, rep)
		if artifacts[env] == nil {
			err = fmt.Errorf("unable to find %v/%v artifact for %v", rep.Owner, rep.Name, env)
		}
		if err != nil {
			config.JSONError(w, 500, "error finding artifact", err)
			return
		}
		artifact = artifacts[env]
	}
	reader, err := downloadReader(artifact)
	if err != nil {
		config.JSONError(w, 500, "error executing download_reader", err)
		return
	}
	defer reader.Close()

	w.Header().Set("Content-type", "application/octet-stream")
	w.Header().Set("Content-disposition", fmt.Sprintf("attachment; filename=\"%v\"", path.Base(artifact.DownloadURL)))
	io.Copy(w, reader)
}
