package artifact

import (
	"context"
	"fmt"
	"io"
	"strings"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/repo"
	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/rlmcpherson/s3gof3r"
)

const (
	envNotFoundPrefix = "Could not find deployed-version in consul "
	DeployBucket      = "core-devtools-deploy-artifact"
)

type DeployedArtifact struct {
	GithubCommit *github.RepositoryCommit `json:"github_commit"`
	Deployment   *deployment.Deployment   `json:"deployment"`
	Artifact     *Artifact                `json:"artifact"`
	Latest       string                   `json:"latest"`
}

type Artifact struct {
	Repository  *repo.Repository
	SHA         string
	DownloadURL string
}

type CompareDeployed struct {
	GithubCompare *github.CommitsComparison `json:"github_compare"`
	Ref           string                    `json:"ref"`
	PreviousSHA   string                    `json:"previous_sha"`
}

func LoadDeployedArtifacts(consulClient *consulapi.Client, githubClient *github.Client, r *repo.Repository) (map[string]*DeployedArtifact, error) {
	deployed := make(map[string]*DeployedArtifact)

	artifacts, err := readDeployedEnvironments(consulClient, r)
	if err != nil {
		return nil, err
	}

	latestByBranch := make(map[string]string)

	for env, artifact := range artifacts {
		commit, _, err := githubClient.Repositories.GetCommit(context.TODO(), r.Owner, r.Name, artifact.SHA)
		if err != nil {
			log.Print(err)
			message := "SHA not found"
			commit = &github.RepositoryCommit{
				SHA: &artifact.SHA,
				Commit: &github.Commit{
					Message: &message,
				},
			}
		}

		deployment, err := deployment.GetCurrentDeploymentForEnviroment(db, r, githubClient, env)
		if err != nil {
			log.Print(err)
			continue
		}

		var branch string
		if deployment != nil {
			branch = *deployment.Branch
		}

		latest := latestByBranch[branch]
		// If latest wasn't cached and we know the branch
		if latest == "" && branch != "" {
			latest, err = latestCommit(githubClient, r.Owner, r.Name, branch)
			if err != nil {
				log.Print("unable to get branch to compare deployment commit to head: ", err)
			} else {
				latestByBranch[branch] = latest
			}
		}

		d := &DeployedArtifact{
			GithubCommit: commit,
			Deployment:   deployment,
			Artifact:     artifact,
			Latest:       latest,
		}

		deployed[env] = d
	}

	return deployed, nil
}

func latestCommit(githubClient *github.Client, owner, name, branch string) (latest string, err error) {
	var gb *github.Branch
	gb, _, err = githubClient.Repositories.GetBranch(context.TODO(), owner, name, branch)
	if err == nil {
		latest = *gb.Commit.SHA
	}

	return
}

func LoadDeployedArtifact(consulClient *consulapi.Client, githubClient *github.Client, r *repo.Repository, env string) (*DeployedArtifact, error) {
	artifact, err := readDeployedEnvironment(consulClient, r, env)
	if err != nil {
		return nil, err
	}

	commit, _, err := githubClient.Repositories.GetCommit(context.TODO(), r.Owner, r.Name, artifact.SHA)
	if err != nil {
		log.Print(err)
		message := "SHA not found"
		commit = &github.RepositoryCommit{
			SHA: &artifact.SHA,
			Commit: &github.Commit{
				Message: &message,
			},
		}
	}

	deployment, err := deployment.GetCurrentDeploymentForEnviroment(db, r, githubClient, env)
	if err != nil {
		log.Print(err)
	}

	var latest string
	if deployment != nil {
		latest, err = latestCommit(githubClient, r.Owner, r.Name, *deployment.Branch)
		if err != nil {
			log.Print("unable to get branch to compare deployment commit to head: ", err)
		}
	}

	d := &DeployedArtifact{
		GithubCommit: commit,
		Deployment:   deployment,
		Artifact:     artifact,
		Latest:       latest,
	}

	log.WithFields(log.Fields{"func": "LoadDeployedArtifact"}).Debugf("DeployedArtifact=%v", d)

	return d, nil

}

func readDeployedEnvironments(client *consulapi.Client, r *repo.Repository) (map[string]*Artifact, error) {
	pairs, _, err := client.KV().List(fmt.Sprintf("%v/%v/%v/", deployment.ConsulPrefixDeployedVersion, r.Owner, r.Name), nil)
	if err != nil {
		return nil, err
	}

	artifacts := make(map[string]*Artifact)
	for _, kv := range pairs {
		parts := strings.Split(kv.Key, "/")
		env := parts[len(parts)-1]

		artifacts[env] = &Artifact{
			Repository:  r,
			SHA:         string(kv.Value),
			DownloadURL: fmt.Sprintf("s3://%v/%v/%v/%v.tgz", DeployBucket, r.Owner, r.Name, string(kv.Value)),
		}
	}

	return artifacts, nil
}

func readDeployedEnvironment(client *consulapi.Client, r *repo.Repository, env string) (*Artifact, error) {
	consul_key := fmt.Sprintf("%v/%v/%v/%v", deployment.ConsulPrefixDeployedVersion, r.Owner, r.Name, env)
	log.WithFields(log.Fields{
		"func":       "readDeployedEnvironment",
		"owner":      r.Owner,
		"name":       r.Name,
		"env":        env,
		"consul_key": consul_key,
	}).Debug("Retrieving SHA from consul")
	kv, _, err := client.KV().Get(consul_key, nil)
	if err != nil {
		return nil, fmt.Errorf("Error retrieving deployed-version in consul (owner=%v, repo=%v, env=%v, key=%v): err=%v", r.Owner, r.Name, env, consul_key, err)
	}
	if kv == nil {
		return nil, newEnvNotFoundError(r.Owner, r.Name, env)
	}
	sha := string(kv.Value)

	artifact := &Artifact{
		Repository:  r,
		SHA:         sha,
		DownloadURL: fmt.Sprintf("s3://%v/%v/%v/%v.tgz", DeployBucket, r.Owner, r.Name, sha),
	}
	log.WithFields(log.Fields{"func": "readDeployedEnvironment"}).Debugf("r.Owner=%v r.Name=%v env=%v artifact.SHA=%v", r.Owner, r.Name, env, artifact.SHA)

	return artifact, nil
}

func isEnvNotFoundError(err error) bool {
	return strings.HasPrefix(err.Error(), "Could not find deployed-version in consul")
}

func newEnvNotFoundError(owner, name, env string) error {
	return fmt.Errorf("%s (owner=%v, repo=%v, env=%v)", envNotFoundPrefix, owner, name, env)
}

func downloadReader(artifact *Artifact) (io.ReadCloser, error) {
	keys, err := s3gof3r.EnvKeys()
	if err != nil {
		keys, err = s3gof3r.InstanceKeys()
		if err != nil {
			return nil, err
		}
	}
	s3 := s3gof3r.New("s3-us-west-2.amazonaws.com", keys)
	bucket := s3.Bucket(DeployBucket)

	s3Path := fmt.Sprintf("%v/%v/%v.tgz", artifact.Repository.Owner, artifact.Repository.Name, artifact.SHA)

	s3gof3r.DefaultConfig.Md5Check = false
	reader, _, err := bucket.GetReader(s3Path, nil)
	if err != nil {
		return nil, err
	}

	return reader, nil
}
