package build

import (
	"context"
	"fmt"
	"net/url"
	"regexp"
	"strconv"
	"time"

	"code.justin.tv/dta/skadi/pkg/repo"
	"code.justin.tv/release/jenkins-api"
	"github.com/google/go-github/github"
)

var (
	jenkinsPathRegexp = regexp.MustCompile("^/job/(.+)/(\\d+)")
)

type Build struct {
	Duration          int `json:"duration"`
	EstimatedDuration int `json:"estimated_duration"`
	Percent           int `json:"percent"`
}

func GetBuild(githubClient *github.Client, jenkinsClient *jenkins.Client, repository *repo.Repository, sha string) (*Build, error) {
	statuses, _, err := githubClient.Repositories.ListStatuses(context.TODO(), repository.Owner, repository.Name, sha, nil)
	if err != nil {
		return nil, err
	}

	if len(statuses) == 0 {
		return nil, nil
	}

	status := statuses[0]

	if status.TargetURL == nil {
		return nil, nil
	}

	job, buildId, err := ParseTargetUrl(*status.TargetURL)
	if err != nil {
		return nil, err
	}

	build, err := jenkinsClient.GetBuild(job, buildId)
	if err != nil {
		return nil, err
	}

	duration, percent := CalculateJenkinsPercent(build)

	b := &Build{
		Duration:          build.Duration,
		EstimatedDuration: build.EstimatedDuration,
		Percent:           int(percent),
	}

	if b.Duration == 0 {
		b.Duration = int(duration)
	}

	return b, nil
}

func CalculateJenkinsPercent(build *jenkins.Build) (float64, float64) {
	var duration float64
	var percent float64

	duration = float64((int64(time.Now().Unix()) * 1000) - build.Timestamp)
	percent = (duration / float64(build.EstimatedDuration)) * 100
	if percent > 100 {
		percent = 100
	}

	return duration, percent
}

func ParseTargetUrl(loc string) (string, int, error) {
	u, err := url.Parse(loc)
	if err != nil {
		return "", 0, fmt.Errorf("invalid jenkins target url. %s", loc)
	}

	matches := jenkinsPathRegexp.FindSubmatch([]byte(u.Path))
	if len(matches) == 0 {
		return "", 0, fmt.Errorf("invalid jenkins target url. %v", loc)
	}

	buildId, err := strconv.Atoi(string(matches[2]))
	if err != nil {
		return "", 0, fmt.Errorf("error parsing build id: %v", err)
	}

	return string(matches[1]), buildId, nil
}
