package candidate

import (
	"context"
	"encoding/json"
	"errors"
	"net/http"

	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"code.justin.tv/dta/skadi/pkg/repo"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"io/ioutil"
	"net/url"
	"strings"
)

var (
	consulClient *consulapi.Client
)

func RegisterHandlers(r *mux.Router, cClient *consulapi.Client) {
	consulClient = cClient

	config.CreateHandler(r, "/v1/candidate/info",
		InfoHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
	config.CreateHandler(r, "/v1/candidate/current",
		CurrentHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
	config.CreateHandler(r, "/v1/candidate/previous",
		PreviousHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
	config.CreateHandler(r, "/v1/candidate/rebuild",
		noopCommitHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("POST")
	config.CreateHandler(r, "/v1/candidate/notifycommit",
		notifyCommitHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("POST")
}

func InfoHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	repository := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}

	client, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, 500, "issue fetching candidate", err)
		return
	}

	repo, _, err := client.Repositories.Get(context.TODO(), repository.Owner, repository.Name)
	if err != nil {
		config.JSONError(w, 500, "issue fetching repository information", err)
		return
	}
	json.NewEncoder(w).Encode(repo)
}

func CurrentHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	repository := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}

	client, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, 500, "issue fetching candidate", err)
		return
	}

	deployConfig, err := config.LoadDeployConfig(
		client,
		consulClient,
		repository.Owner,
		repository.Name,
		githubcache.GetDefaultBranch(repository.Owner, repository.Name),
	)
	if err != nil {
		config.JSONError(w, 500, "issue fetching deploy config", err)
		return
	}

	// query branch list for repo, collect candidate state from each tip's commit status
	candidates, err := LoadCurrent(client, repository, deployConfig)
	if err != nil {
		config.JSONError(w, 500, "issue fetching candidate", err)
		return
	}
	json.NewEncoder(w).Encode(candidates)
}

func PreviousHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	// given a branch, show a commit-sorted list of previous candidates
	repository := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}

	client, err := config.GithubClient(ctx, true)
	if err != nil {
		config.JSONError(w, 500, "issue fetching candidate", err)
		return
	}

	deployConfig, err := config.LoadDeployConfig(
		client,
		consulClient,
		repository.Owner,
		repository.Name,
		githubcache.GetDefaultBranch(repository.Owner, repository.Name),
	)
	if err != nil {
		config.JSONError(w, 500, "issue fetching deploy config", err)
		return
	}

	// use given branch to query github for recent commits
	// transform statuses to candidate, keeping descending commit order
	candidates, err := LoadPrevious(client, repository, r.FormValue("branch"), deployConfig)
	if err != nil {
		config.JSONError(w, 500, "issue fetching previous candidates", err)
		return
	}
	json.NewEncoder(w).Encode(candidates)
}

func noopCommitHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	repository := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}

	urlQuery := r.URL.Query()

	ref := urlQuery.Get("ref")
	if ref == "" {
		config.JSONError(w, http.StatusBadRequest, "issue processing rebuild", errors.New("ref is required"))
		return
	}

	client, err := config.GithubClient(ctx, true)
	if err != nil {
		config.JSONError(w, 500, "issue processing rebuild", err)
		return
	}

	if err := Rebuild(client, repository, ref); err != nil {
		config.JSONError(w, 500, "issue processing rebuild", err)
		return
	}
}

func notifyCommitHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	repository := &repo.Repository{Owner: r.FormValue("owner"), Name: r.FormValue("name")}
	ref := r.FormValue("ref")
	if ref == "" {
		config.JSONError(w, http.StatusBadRequest, "Invalid arguments", errors.New("ref is required"))
		return
	}

	client, err := config.JenkinsClient(ctx)
	if err != nil {
		config.JSONError(w, 500, "failed to get jenkins client", err)
		return
	}

	query := &url.Values{}
	query.Add("url", fmt.Sprintf("git@git.xarth.tv:%s/%s.git", repository.Owner, repository.Name))
	query.Add("sha1", ref)
	req, err := client.NewRequest("GET", "/git/notifyCommit", nil, query)
	if err != nil {
		config.JSONError(w, 500, "failed to create a request", err)
		return
	}

	res, err := http.DefaultClient.Do(req)
	if err != nil {
		config.JSONError(w, 500, "failed to send a request", err)
		return
	}
	defer res.Body.Close()

	// invalidate the cached commit status
	deleteStatusCache(repository, ref)

	body, err := ioutil.ReadAll(res.Body)
	if err != nil {
		config.JSONError(w, 500, "failed to parse the response body", err)
		return
	}
	log.Printf("notifyCommit: %+v/%s - %s", repository, ref, string(body))

	if !strings.HasPrefix(string(body), "Scheduled") {
		config.JSONError(w, 403, "failed to schedule a build - Has 'Poll SCM' trigger been enabled on the Jenkins job?", err)
		return
	}
}
