package candidate

import (
	"context"
	"fmt"
	"sort"
	"strings"
	"sync"
	"time"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/dta/skadi/pkg/build"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"code.justin.tv/dta/skadi/pkg/repo"
	"code.justin.tv/dta/skadi/pkg/user"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/google/go-github/github"
	"github.com/pmylund/go-cache"
)

const (
	// The data in commit and status caches is kind of permanent immutable data,
	// so they don't need to be expired but we want the caches to use the memory
	// effectively and to be filled with likely reused data.
	cacheCleanupInterval       = 5 * time.Minute
	commitCacheExpiration      = 24 * time.Hour
	statusCacheExpiration      = 24 * time.Hour
	statusCacheExpirationShort = 10 * time.Minute
	statusCacheFillOlderThan   = 30 * time.Minute
)

var (
	commitCache = cache.New(commitCacheExpiration, cacheCleanupInterval)
	statusCache = cache.New(statusCacheExpiration, cacheCleanupInterval)
)

type Candidate struct {
	SHA    *string    `json:"sha,omitempty"`
	Branch *string    `json:"branch,omitempty"`
	Date   *time.Time `json:"date,omitempty"`
	Author *user.User `json:"author,omitempty"`

	State       *string            `json:"state,omitempty"`
	States      *map[string]string `json:"states,omitempty"`
	TargetURL   *string            `json:"target_url,omitempty"`
	Description *string            `json:"description,omitempty"`

	Commit *github.Commit `json:"commit,omitempty"`
}

type StateMetadata struct {
	State       *string
	TargetURL   *string
	Description *string
	Date        *time.Time
}

func NewCandidate(branch string, commit *github.RepositoryCommit, status *StateMetadata, statuses *map[string]StateMetadata) *Candidate {
	state := "unknown"
	states := map[string]string{}

	candidate := &Candidate{
		SHA:    commit.SHA,
		Branch: &branch,
		Date:   commit.Commit.Author.Date,
		Commit: commit.Commit,
		Author: user.NewUser(commit.Author),
		State:  &state,
		States: &states,
	}

	if status != nil {
		candidate.State = status.State
		candidate.TargetURL = status.TargetURL
		candidate.Description = status.Description
		candidate.Date = status.Date
	}

	if statuses != nil {
		for environment, s := range *statuses {
			(*candidate.States)[environment] = *s.State
		}
	}
	return candidate
}

// load branch list from github, then commit statuses for each branch
// list sorted by master always first, then recent commits first
func LoadCurrent(client *github.Client, repository *repo.Repository, deployConfig *api.DeployConfig) ([]*Candidate, error) {
	heads, _, err := client.Git.ListRefs(context.TODO(), repository.Owner, repository.Name, &github.ReferenceListOptions{Type: "heads"})
	if err != nil {
		return nil, err
	}

	var wg sync.WaitGroup
	candidateChan := make(chan *Candidate, len(heads))
	errChan := make(chan error, len(heads))
	limiter := make(chan struct{}, 10)

	for _, head := range heads {
		wg.Add(1)

		branch := strings.Replace(*head.Ref, "refs/heads/", "", 1)

		go func(branch, sha string) {
			defer wg.Done()

			limiter <- struct{}{}
			defer func() {
				<-limiter
			}()

			commit, statusMetadata, statuses, err := loadCandidateGithubData(client, repository, sha, deployConfig)
			if err != nil {
				errChan <- err
				return
			}

			candidateChan <- NewCandidate(branch, commit, statusMetadata, statuses)
		}(branch, *head.Object.SHA)
	}

	wg.Wait()
	close(candidateChan)

	if len(errChan) > 0 {
		return nil, <-errChan
	}

	var candidates []*Candidate
	var master *Candidate

	defaultBranch := githubcache.GetDefaultBranch(repository.Owner, repository.Name)
	for candidate := range candidateChan {
		if *candidate.Branch == defaultBranch {
			master = candidate
		} else {
			candidates = append(candidates, candidate)
		}
	}

	sort.Sort(ByDate(candidates))

	candidates = append([]*Candidate{master}, candidates...)

	return candidates, nil
}

type ByDate []*Candidate

func (c ByDate) Len() int           { return len(c) }
func (c ByDate) Swap(i, j int)      { c[i], c[j] = c[j], c[i] }
func (c ByDate) Less(i, j int) bool { return (*c[i].Date).After(*c[j].Date) }

func LoadCandidateBuildStatus(client *github.Client, repository *repo.Repository, sha string) (*build.Build, error) {
	return nil, nil
}

func getRepositoryCommit(client *github.Client, repository *repo.Repository, sha string) (*github.RepositoryCommit, error) {
	logger := log.WithFields(log.Fields{"function": "getRepositoryCommit", "repository": repository.FullName(), "sha": sha})
	cacheKey := fmt.Sprintf("%s/%s", repository.FullName(), sha)
	if commit, ok := commitCache.Get(cacheKey); ok {
		logger.Debug("CACHE HIT")
		return commit.(*github.RepositoryCommit), nil
	}
	logger.Debug("CACHE MISS")
	commit, _, err := client.Repositories.GetCommit(context.TODO(), repository.Owner, repository.Name, sha)
	if err != nil {
		return nil, err
	}
	commitCache.Set(cacheKey, commit, 0)
	return commit, nil
}

func statusToStateMetadata(status *github.RepoStatus) *StateMetadata {
	var sm StateMetadata
	if status == nil {
		return nil
	}
	if status.State != nil {
		sm.State = aws.String(*status.State)
	}
	if status.TargetURL != nil {
		sm.TargetURL = aws.String(*status.TargetURL)
	}
	if status.Description != nil {
		sm.Description = aws.String(*status.Description)
	}
	if status.UpdatedAt != nil {
		sm.Date = aws.Time(*status.UpdatedAt)
	} else if status.CreatedAt != nil {
		sm.Date = aws.Time(*status.CreatedAt)
	}
	return &sm
}

func getRepoStateMetadata(client *github.Client, repository *repo.Repository, sha string, deployConfig *api.DeployConfig) (*StateMetadata, error) {
	var ret *StateMetadata
	logger := log.WithFields(log.Fields{"function": "getRepoStateMetadata", "repository": repository.FullName(), "sha": sha})
	cacheKey := fmt.Sprintf("%s/%s", repository.FullName(), sha)
	if status, ok := statusCache.Get(cacheKey); ok {
		logger.Debug("CACHE HIT")
		ret = status.(*StateMetadata)
	} else {
		logger.Debug("CACHE MISS")
		statuses, _, err := client.Repositories.ListStatuses(context.TODO(), repository.Owner, repository.Name, sha, nil)
		if err != nil {
			return nil, err
		}
		ret = getStateMetadataAndFillStatusCache(statuses, deployConfig, logger, cacheKey)
	}
	return ret, nil
}

func deleteStatusCache(repository *repo.Repository, sha string) {
	cacheKey := fmt.Sprintf("%s/%s", repository.FullName(), sha)
	statusCache.Delete(cacheKey)
}

func cacheKeyForRepoStatusWithEnv(repository *repo.Repository, sha, envName string) string {
	return fmt.Sprintf("%s/%s/%s", repository.FullName(), sha, envName)
}

func getRepoStateMetadataMap(client *github.Client, repository *repo.Repository, sha string, deployConfig *api.DeployConfig) (*map[string]StateMetadata, error) {
	repoStatuses := map[string]StateMetadata{}
	var uncachedStatuses []string
	logger := log.WithFields(log.Fields{"function": "getRepoStateMetadataMap", "repository": repository.FullName(), "sha": sha})
	if deployConfig.Environments != nil {
		for envName := range *deployConfig.Environments {
			cacheKey := cacheKeyForRepoStatusWithEnv(repository, sha, envName)
			logger.Data["environment"] = envName
			if status, ok := statusCache.Get(cacheKey); ok {
				logger.Debug("CACHE HIT")
				repoStatuses[envName] = *status.(*StateMetadata)
			} else {
				uncachedStatuses = append(uncachedStatuses, envName)
				logger.Debug("CACHE MISS")
			}
		}
		if len(uncachedStatuses) > 0 {
			statuses, _, err := client.Repositories.ListStatuses(context.TODO(), repository.Owner, repository.Name, sha, nil)
			if err != nil {
				return nil, err
			}
			for _, envName := range uncachedStatuses {
				cacheKey := cacheKeyForRepoStatusWithEnv(repository, sha, envName)
				logger.Data["environment"] = envName
				status := getStateMetadataAndFillStatusCacheWithEnv(statuses, deployConfig, logger, cacheKey, envName)
				if status != nil {
					repoStatuses[envName] = *status
				}
			}
		}
	}
	return &repoStatuses, nil
}

// Cache status only when it's older than 30 minutes. Teams often have other jobs that change commit
// statuses outside of the first initial job.
func cacheStateMetadata(sm *StateMetadata, logger *log.Entry, cacheKey string) {
	if sm.Date != nil && time.Now().UTC().Sub(*sm.Date) >= statusCacheFillOlderThan {
		logger.Data["state"] = sm.State
		if *sm.State == "success" || *sm.State == "aborted" {
			logger.Debug("CACHE FILL")
			statusCache.Set(cacheKey, sm, 0)
		} else if *sm.State == "failure" || *sm.State == "error" {
			logger.Debug("CACHE FILL(short live)")
			statusCache.Set(cacheKey, sm, statusCacheExpirationShort)
		} else {
			// don't cache any other statuses to make skadi get
			// fresh information
			logger.Debug("CACHE not filled")
		}
	}
}

func getStateMetadataAndFillStatusCache(statuses []*github.RepoStatus, deployConfig *api.DeployConfig, logger *log.Entry, cacheKey string) *StateMetadata {
	sm := getStateMetadata(deployConfig, statuses)
	if sm != nil {
		cacheStateMetadata(sm, logger, cacheKey)
	}
	return sm
}

func getStateMetadataAndFillStatusCacheWithEnv(statuses []*github.RepoStatus, deployConfig *api.DeployConfig, logger *log.Entry, cacheKey, envName string) *StateMetadata {
	var sm *StateMetadata
	sm = getStateMetadataWithEnv(deployConfig, statuses, envName)
	if sm != nil {
		cacheStateMetadata(sm, logger, cacheKey)
	}
	return sm
}

// When a build is aborted, jenkins reports the status to github as "pending".
// If the current status is pending, and the next status is pending, then this sha was aborted
// and mark it as failed.
// https://twitchtv.atlassian.net/browse/DTA-1573
func getStatusAndProcessPending(statuses []*github.RepoStatus, i int) (status *github.RepoStatus) {
	status = statuses[i]
	if status != nil && status.State != nil && *status.State == "pending" && checkPendingArr(statuses, i) {
		*status.State = "aborted"
		if status.Description != nil {
			dPieces := strings.Split(*status.Description, " ")
			// The status description is "Build #_number_ in progress...". Use the first two words and append aborted.
			*status.Description = strings.Join(append(dPieces[:2], "aborted"), " ")
		} else {
			*status.Description = "aborted"
		}
	}
	return
}

func checkPendingArr(statuses []*github.RepoStatus, i int) bool {
	// If next status exceeds array boundaries
	if len(statuses) <= i+1 {
		return false
	}

	return checkPending(statuses[i], statuses[i+1])
}

func checkPending(current, next *github.RepoStatus) bool {
	if next.State == nil || current.State == nil {
		return false
	}

	if next.Context == nil || current.Context == nil {
		return false
	}

	return *current.State == "pending" && *next.State == "pending" && *next.Context == *current.Context
}

func getStateMetadata(deployConfig *api.DeployConfig, statuses []*github.RepoStatus) *StateMetadata {
	if deployConfig.RequiredContexts != nil {
		return getStateMetadataFromContexts(*deployConfig.RequiredContexts, statuses, false)
	}
	return getStateMetadataFromContext(deployConfig, statuses)
}

func getStateMetadataWithEnv(deployConfig *api.DeployConfig, statuses []*github.RepoStatus, environment string) *StateMetadata {
	if deployConfig.Environments != nil {
		appEnv, hasEnvironmentConfig := (*deployConfig.Environments)[environment]
		if hasEnvironmentConfig && appEnv.RequiredContexts != nil {
			return getStateMetadataFromContexts(*appEnv.RequiredContexts, statuses, true)
		}
	}
	if deployConfig.RequiredContexts != nil {
		return getStateMetadataFromContexts(*deployConfig.RequiredContexts, statuses, true)
	}
	return getStateMetadataFromContext(deployConfig, statuses)
}

// getStateMetadataFromContexts verifies that all required contexts are succesful
// If forceEmptySuccess is true and contexts is an empty list, then the
// first status will be returned with its state changed to "success"
func getStateMetadataFromContexts(contexts []string, statuses []*github.RepoStatus, forceEmptySuccess bool) *StateMetadata {
	var status, firstRequiredStatus *github.RepoStatus
	var sm *StateMetadata
	successfulRequired := map[string]*github.RepoStatus{}
	for i := range statuses {
		status = getStatusAndProcessPending(statuses, i)
		if len(contexts) == 0 {
			sm = statusToStateMetadata(status)
			if forceEmptySuccess {
				// HACK: required contexts is an empty list, so we just return the
				// first status and force it to be successful
				*sm.State = "success"
			}
			return sm
		}
		for _, ctx := range contexts {
			if ctx == *status.Context {
				_, found := successfulRequired[*status.Context]
				if !found {
					// this is the first time we see this context in the iteration,
					// which means it's the most recent one and is the status of the context
					if *status.State == "success" {
						if firstRequiredStatus == nil {
							firstRequiredStatus = status
						}
						successfulRequired[*status.Context] = status
					} else {
						// we return on the first mandatory required context that isn't `success`
						return statusToStateMetadata(status)
					}
				}
			}
		}
	}
	if len(contexts) == len(successfulRequired) {
		// verify that all required contexts were successful
		return statusToStateMetadata(firstRequiredStatus)
	}
	return nil
}

func getStateMetadataFromContext(deployConfig *api.DeployConfig, statuses []*github.RepoStatus) *StateMetadata {
	var status *github.RepoStatus
	for i := range statuses {
		status = getStatusAndProcessPending(statuses, i)
		if deployConfig.BuildStatusContext == nil || *deployConfig.BuildStatusContext == *status.Context {
			return statusToStateMetadata(status)
		}
	}
	return nil
}

func loadCandidateGithubData(client *github.Client, repository *repo.Repository, sha string, deployConfig *api.DeployConfig) (commit *github.RepositoryCommit, sm *StateMetadata, sms *map[string]StateMetadata, err error) {
	commit, err = getRepositoryCommit(client, repository, sha)
	if err == nil {
		sm, err = getRepoStateMetadata(client, repository, sha, deployConfig)
	}
	if err == nil {
		sms, err = getRepoStateMetadataMap(client, repository, sha, deployConfig)
	}
	return
}

// LoadPrevious takes a branch and loads recent candidates on that branch.  It uses the most recent commit status
// on each commit to determine the status of the candidate.
// https://developer.github.com/v3/repos/statuses/
func LoadPrevious(client *github.Client, repository *repo.Repository, branch string, deployConfig *api.DeployConfig) ([]*Candidate, error) {
	var candidates []*Candidate

	commits, _, err := client.Repositories.ListCommits(context.TODO(), repository.Owner, repository.Name, &github.CommitsListOptions{SHA: branch})
	if err != nil {
		return nil, err
	}

	for _, commit := range commits {
		var sm *StateMetadata
		statuses, _, err := client.Repositories.ListStatuses(context.TODO(), repository.Owner, repository.Name, *commit.SHA, nil)
		if err != nil {
			return nil, err
		}
		if len(statuses) > 0 {
			sm = getStateMetadata(deployConfig, statuses)
		}

		log.WithFields(log.Fields{
			"function": "LoadPrevious",
			"branch":   branch,
			"sm":       sm,
			"commit":   commit,
		}).Info("Retrieving Candidate")

		candidate := NewCandidate(branch, commit, sm, nil)
		log.WithFields(log.Fields{
			"function":  "LoadPrevious",
			"candidate": candidate,
		}).Info("Got Candidate")

		candidates = append(candidates, candidate)
	}

	return candidates, nil
}

// Rebuild triggers a rebuild of a jenkins job given the git ref and target_url from
// the github commit status API.
func Rebuild(githubClient *github.Client, repository *repo.Repository, ref string) error {
	owner := repository.Owner
	repo := repository.Name

	ref = "heads/" + ref

	// Get latest sha for ref
	existingRef, _, err := githubClient.Git.GetRef(context.TODO(), owner, repo, ref)
	if err != nil {
		return err
	}
	sha := *existingRef.Object.SHA

	// Get latest commit for ref
	latestCommit, _, err := githubClient.Git.GetCommit(context.TODO(), owner, repo, sha)
	if err != nil {
		log.Fatal("unable to get tree:", err)
	}

	// Get the authenticated user
	var user github.User
	req, err := githubClient.NewRequest("GET", "user", nil)
	if err != nil {
		return err
	}
	_, err = githubClient.Do(context.TODO(), req, &user)
	if err != nil {
		return err
	}

	// Create a commit message
	message := fmt.Sprintf("noop commit %s\n\nCreated by skadi to retry building %s.",
		sha, sha)

	// Author is devtools
	author := github.CommitAuthor{
		Name:  user.Name,
		Email: strPtr(*user.Login + "@justin.tv"),
	}

	// Build the commit
	newCommit := github.Commit{
		Author:    &author,
		Committer: &author,
		Message:   &message,
		Tree:      latestCommit.Tree,
		Parents: []github.Commit{
			{
				SHA: &sha,
			},
		},
	}

	// Create the commit
	noop, _, err := githubClient.Git.CreateCommit(context.TODO(), owner, repo, &newCommit)
	if err != nil {
		return err
	}

	// Update ref with new commit
	_, _, err = githubClient.Git.UpdateRef(context.TODO(), owner, repo, &github.Reference{
		Ref: &ref,
		Object: &github.GitObject{
			SHA: noop.SHA,
		},
	}, false)

	return err
}

func strPtr(str string) *string {
	return &str
}
