package candidate

import (
	"testing"

	"code.justin.tv/dta/skadi/api"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/google/go-github/github"
)

func TestGetStatusAndFillStatusCache(t *testing.T) {
	tests := []struct {
		name                string
		statuses            []*github.RepoStatus
		deployConfig        *api.DeployConfig
		expectedStatus      string
		expectedDescription string
	}{
		{
			name: "nil contexts shouldn't panic",
			statuses: []*github.RepoStatus{
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
				},
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
				},
			},
			expectedStatus: "pending",
			deployConfig:   &api.DeployConfig{},
		},
		{
			name: "two pending statuses from different contexts + success == success",
			statuses: []*github.RepoStatus{
				{
					Description: aws.String("Build #1 succeeded"),
					State:       aws.String("success"),
					Context:     aws.String("4321"),
				},
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
					Context:     aws.String("4321"),
				},
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
					Context:     aws.String("1234"),
				},
			},
			expectedStatus: "success",
			deployConfig:   &api.DeployConfig{},
		},
		{
			name: "two pending statuses from different contexts == pending",
			statuses: []*github.RepoStatus{
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
					Context:     aws.String("4321"),
				},
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
					Context:     aws.String("1234"),
				},
			},
			expectedStatus: "pending",
			deployConfig:   &api.DeployConfig{},
		},
		{
			name: "two pending statuses makes a failure",
			statuses: []*github.RepoStatus{
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
					Context:     aws.String("1234"),
				},
				{
					Description: aws.String("Build #1 in progress..."),
					State:       aws.String("pending"),
					Context:     aws.String("1234"),
				},
			},
			expectedStatus:      "aborted",
			expectedDescription: "Build #1 aborted",
			deployConfig:        &api.DeployConfig{},
		},
		{
			name: "success status",
			statuses: []*github.RepoStatus{
				{
					State: aws.String("success"),
				},
				{
					State: aws.String("pending"),
				},
			},
			expectedStatus: "success",
			deployConfig:   &api.DeployConfig{},
		},
		{
			name: "one pending status should be returned",
			statuses: []*github.RepoStatus{
				{
					State: aws.String("pending"),
				},
			},
			expectedStatus: "pending",
			deployConfig:   &api.DeployConfig{},
		},
	}

	logger := log.WithFields(log.Fields{"test": "TestGetStatusAndFillStatusCache"})

	for _, sample := range tests {
		t.Run(sample.name, func(t *testing.T) {
			status := getStateMetadataAndFillStatusCache(sample.statuses, sample.deployConfig, logger, "anykey")
			if *status.State != sample.expectedStatus {
				t.Errorf("%s: status state %q != sample status %q", sample.name, *status.State, sample.expectedStatus)
			}

			if sample.expectedDescription != "" && *status.Description != sample.expectedDescription {
				t.Errorf("%s: status description %q != sample description %q", sample.name, *status.Description, sample.expectedDescription)
			}

		})
	}
}
