package config

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
)

// LoadChatConfig finds a ChatConfig from the github repository on the given ref and sets default values where missing
func LoadChatConfig(githubClient *github.Client, consulClient *consulapi.Client, owner, name, ref string) (*api.ChatConfig, error) {
	config, err := loadChatConfigFile(githubClient, consulClient, owner, name, ref)
	if err != nil {
		return nil, err
	}
	setChatConfigDefaults(config, githubcache.GetDefaultBranch(owner, name))
	return config, nil
}

// LoadChatConfig given a ref finds a ChatConfig in deploy.json, slack.json in the github repository, or returns nil
func loadChatConfigFile(githubClient *github.Client, consulClient *consulapi.Client, owner, name, ref string) (*api.ChatConfig, error) {
	c, err := LoadDeployConfig(githubClient, consulClient, owner, name, ref)
	if err == nil && c.Slack != nil {
		return c.Slack, nil
	}

	file, _, resp, err := githubClient.Repositories.GetContents(context.TODO(), owner, name, "slack.json", &github.RepositoryContentGetOptions{Ref: ref})
	if err != nil {
		if resp != nil && resp.Response.StatusCode == http.StatusNotFound { // not found is ok, other status codes indicate error
			return &api.ChatConfig{}, nil
		}
		return nil, err
	}

	data, err := base64.StdEncoding.DecodeString(*file.Content)
	if err != nil {
		return nil, err
	}

	var config api.ChatConfig
	err = json.NewDecoder(bytes.NewBuffer(data)).Decode(&config)
	if err != nil {
		return nil, err
	}

	if config.Channel == nil && config.NotifyDeployUser != true {
		return nil, fmt.Errorf("LoadChatConfig: missing channel field for %v/%v@%v", owner, name, ref)
	}
	return &config, nil
}

func setChatConfigDefaults(config *api.ChatConfig, defaultBranch string) {
	// by default, only broadcast build results on the default branch
	if config.Build == nil {
		branches := map[string]api.ChatSectionConfig{
			defaultBranch: {
				Pending: false,
				Success: true,
				Failure: true,
			},
		}
		config.Build = &api.ChatBranchesConfig{
			Branches: branches,
		}
	}

	// by default, only broadcast coverage issues in the default branch
	if config.Coverage == nil {
		branches := map[string]api.ChatSectionConfig{
			defaultBranch: {
				Pending: false,
				Success: false,
				Failure: true,
			},
		}
		config.Coverage = &api.ChatBranchesConfig{
			Branches: branches,
		}
	}

	// by default, broadcast all deploy results for all envs
	if config.Deploy == nil {
		envs := map[string]api.ChatSectionConfig{
			"*": {
				Pending: false,
				Success: true,
				Failure: true,
			},
		}

		config.Deploy = &api.ChatEnvironmentsConfig{
			Environments: envs,
		}
	}
}
