package config

import (
	"encoding/base64"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/dta/skadi/api"
	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
)

func setupGithub() (*http.ServeMux, *httptest.Server, *github.Client) {
	mux := http.NewServeMux()
	server := httptest.NewServer(mux)

	client := github.NewClient(nil)
	url, _ := url.Parse(server.URL + "/")
	client.BaseURL = url

	return mux, server, client
}

func setupConsul() (*http.ServeMux, *httptest.Server, *consulapi.Client) {
	mux := http.NewServeMux()
	server := httptest.NewServer(mux)

	conf := consulapi.DefaultConfig()
	url, _ := url.Parse(server.URL)
	conf.Address = url.Host
	client, _ := consulapi.NewClient(conf)

	return mux, server, client
}

func TestChatConfigLegacyFile(t *testing.T) {
	mux, server, client := setupGithub()
	defer server.Close()
	_, server, consulClient := setupConsul()
	defer server.Close()

	mux.HandleFunc("/repos/release/example/contents/slack.json", func(w http.ResponseWriter, r *http.Request) {
		j := `{"channel": "#legacyfiletest"}`
		encoded := base64.StdEncoding.EncodeToString([]byte(j))
		fmt.Fprint(w, fmt.Sprintf(`{"type": "file", "encoding": "base64", "content": "%v"}`, encoded))
	})
	config, err := LoadChatConfig(client, consulClient, "release", "example", "master")
	if err != nil {
		t.Fatal(err)
	}
	if *config.Channel != "#legacyfiletest" {
		t.Fatalf("incorrect chat channel")
	}
}

func TestChatConfigDeployFile(t *testing.T) {
	mux, server, client := setupGithub()
	defer server.Close()
	_, server, consulClient := setupConsul()
	defer server.Close()

	ClearConfigCache()
	mux.HandleFunc("/repos/release/example/contents/deploy.json", func(w http.ResponseWriter, r *http.Request) {
		j := `{"slack": {"channel": "#deployfiletest"}}`
		encoded := base64.StdEncoding.EncodeToString([]byte(j))
		fmt.Fprint(w, fmt.Sprintf(`{"type": "file", "encoding": "base64", "content": "%v"}`, encoded))
	})
	config, err := LoadChatConfig(client, consulClient, "release", "example", "master")
	if err != nil {
		t.Fatal(err)
	}
	if *config.Channel != "#deployfiletest" {
		t.Fatalf("incorrect chat channel")
	}
}

func TestChatConfigDeployFileWithoutChatSection(t *testing.T) {
	mux, server, client := setupGithub()
	defer server.Close()
	_, server, consulClient := setupConsul()
	defer server.Close()

	ClearConfigCache()
	mux.HandleFunc("/repos/release/example/contents/deploy.json", func(w http.ResponseWriter, r *http.Request) {
		j := `{}`
		encoded := base64.StdEncoding.EncodeToString([]byte(j))
		fmt.Fprint(w, fmt.Sprintf(`{"type": "file", "encoding": "base64", "content": "%v"}`, encoded))
	})
	config, err := LoadChatConfig(client, consulClient, "release", "example", "master")
	if err != nil {
		t.Fatal(err)
	}
	if config.Channel != nil {
		t.Fatalf("chat channel should not be set")
	}
}

func TestChatConfigNoFile(t *testing.T) {
	_, server, client := setupGithub()
	defer server.Close()
	_, server, consulClient := setupConsul()
	defer server.Close()

	_, err := LoadChatConfig(client, consulClient, "release", "example", "master")
	if err != nil {
		t.Fatal(err)
	}
}

func TestChatConfigDefaults(t *testing.T) {
	config := &api.ChatConfig{}
	setChatConfigDefaults(config, "master")
	if !config.Build.Branches["master"].Success {
		t.Fatalf("master branch did not default to success")
	}
	if !config.Build.Branches["master"].Failure {
		t.Fatalf("master branch did not default to failure")
	}
	if !config.Deploy.Environments["*"].Success {
		t.Fatalf("all environments did not default to success")
	}
	if !config.Deploy.Environments["*"].Failure {
		t.Fatalf("all environments did not default to failure")
	}
}

func TestChatConfigCustom(t *testing.T) {
	config := &api.ChatConfig{
		Deploy: &api.ChatEnvironmentsConfig{
			Environments: map[string]api.ChatSectionConfig{
				"production": api.ChatSectionConfig{Pending: true},
			},
		},
	}
	setChatConfigDefaults(config, "master")
	if !config.Deploy.Environments["production"].Pending {
		t.Fatalf("production env not set to pending")
	}
}
