package config

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"path"
	"time"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/dta/skadi/api"
	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/pmylund/go-cache"
	"strings"
)

var (
	configSettingsCache *cache.Cache
)

const (
	configCacheExpiration      = 60 * time.Second
	configCacheCleanupInterval = 180 * time.Second
)

func init() {
	configSettingsCache = cache.New(configCacheExpiration, configCacheCleanupInterval)
}

func ClearConfigCache() {
	configSettingsCache.Flush()
}

// LoadDeployConfig decodes the deploy.json directly from the github repository on the given ref
func LoadDeployConfig(githubClient *github.Client, consulClient *consulapi.Client, owner, name, ref string) (*api.DeployConfig, error) {
	//check if we have it in cache first
	cacheKey := fmt.Sprintf("config/%s/%s@%s", owner, name, ref)
	if config, found := configSettingsCache.Get(cacheKey); found {
		return config.(*api.DeployConfig), nil
	}
	var err error
	config := &api.DeployConfig{}

	err = LoadGithubConfig(config, githubClient, owner, name, ref)
	if err != nil {
		return nil, fmt.Errorf("error loading github settings %q: %v", path.Join(owner, name), err)
	}

	err = LoadConsulConfig(config, consulClient, owner, name, ref)
	if err != nil {
		return nil, fmt.Errorf("error loading consul settings %q: %v", path.Join(owner, name), err)
	}

	configSettingsCache.Set(cacheKey, config, cache.DefaultExpiration)
	return config, nil
}

func LoadConsulConfig(config *api.DeployConfig, consulClient *consulapi.Client, owner, name, ref string) error {
	pairs, _, err := consulClient.KV().List("dynamic-envs/"+owner+"/"+name+"/", nil)
	if err != nil {
		log.Printf("error getting dynamic environments: %v", err)
	}

	for _, pair := range pairs {
		// Sometimes, not all the times, consul returns including the parent directory where it starts
		// to search which is not a valid dynamic environment.
		if strings.HasSuffix(pair.Key, "/") {
			continue
		}
		envname := path.Base(pair.Key)
		log.Debugf("Adding dynamic environment %v, %s", pair.Key, envname)
		_, keyfound := (*config.Environments)[envname]
		if !keyfound {
			(*config.Environments)[envname] = api.AppEnvironmentConfig{}
		}
	}

	return nil
}

func LoadGithubConfig(config *api.DeployConfig, githubClient *github.Client, owner, name, ref string) error {
	file, _, _, err := githubClient.Repositories.GetContents(context.TODO(), owner, name, "deploy.json", &github.RepositoryContentGetOptions{Ref: ref})
	if err != nil {
		return err
	}
	log.WithFields(log.Fields{"func": "LoadGithubConfig"}).Debugf("deploy.json base64 encoded file contents: %v", *file.Content)

	data, err := base64.StdEncoding.DecodeString(*file.Content)
	if err != nil {
		return err
	}

	err = json.NewDecoder(bytes.NewBuffer(data)).Decode(config)
	if err != nil {
		return err
	}

	// Check entries that hasn't set.
	if config.Datacenters == nil {
		config.Datacenters = &[]string{}
	}

	return nil
}

// FindAppEnvironmentConfig loads deploy.json and finds the environments config for the given owner, name, and ref
func FindAppEnvironmentConfig(githubClient *github.Client, consulClient *consulapi.Client, owner, name, ref, env string) (*api.AppEnvironmentConfig, error) {
	config, err := LoadDeployConfig(githubClient, consulClient, owner, name, ref)
	if err != nil {
		return nil, err
	}
	if config.Environments == nil {
		return nil, errors.New("environments not set")
	}
	environment, ok := (*config.Environments)[env]
	if !ok {
		return nil, errors.New("environment not in configuration")
	}
	return &environment, nil
}
