package config

import (
	"encoding/json"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/chat/timing"
	"code.justin.tv/common/chitin"
	"code.justin.tv/dta/skadi/pkg/info"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/gorilla/mux"
	"github.com/gorilla/schema"
	"github.com/stvp/rollbar"
	"golang.org/x/net/context"
)

const (
	// AccessControlOrigin is a header set for CORS.
	AccessControlOrigin = "Access-Control-Allow-Origin"
	// AccessControlHeaders is a header set for CORS.
	AccessControlHeaders = "Access-Control-Allow-Headers"
	// AccessControlCredentials is needed for CORS to work behind AEA
	// See https://w.amazon.com/index.php/NextGenMidway/UserGuide/Upgrading%20the%20AWSSSLProxy#CORS
	AccessControlCredentials = "Access-Control-Allow-Credentials"
)

var (
	// One schemaDecoder should be shared amongst all packages as it caches
	// values per struct type.
	schemaDecoder = schema.NewDecoder()
	dinfo         = info.NewInfo("webapi")
	StatsdClient  statsd.Statter
)

func init() {
	schemaDecoder.IgnoreUnknownKeys(true)
}

// ErrorResponse represents a JSON error.
type ErrorResponse struct {
	Error         string
	VerboseError  string
	StatusCode    int
	StatusMessage string
}

// JSONError will format and return a JSON error. Msg is sent to the user and
// err is logged.
func JSONError(w http.ResponseWriter, statusCode int, msg string, err error) {
	log.Printf("error StatusCode=%d msg=%q err=%q", statusCode, msg, err)
	w.WriteHeader(statusCode)

	if rollbar.Token != "" && statusCode >= 500 {
		rollbar.Message(rollbar.ERR, fmt.Sprintf("%v: %v", msg, err))
	}

	errResponse := &ErrorResponse{
		Error:         msg,
		StatusCode:    statusCode,
		StatusMessage: http.StatusText(statusCode),
	}
	if err != nil {
		errResponse.VerboseError = err.Error()
	}

	if jsonErr := json.NewEncoder(w).Encode(errResponse); jsonErr != nil {
		log.Printf("error msg=%q err=%q", "Error encoding JSON Error.", jsonErr)
	}
}

type ResponseWriter struct {
	http.ResponseWriter
	statusCode int
}

func (w *ResponseWriter) WriteHeader(status int) {
	w.ResponseWriter.WriteHeader(status)
	w.statusCode = status
}

// AddCORSHeaders is a wrapper function to correctly setup headers and handle the
// OPTIONS header needed for CORS.
func AddCORSHeaders(f http.HandlerFunc) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		w.Header().Set(AccessControlOrigin, "*")
		w.Header().Set(AccessControlHeaders, "GithubAccessToken, Content-Type")
		w.Header().Set(AccessControlCredentials, "true")
		if r.Method == "OPTIONS" {
			return
		}

		f(w, r)
	}
}

// ParseReq is a small helper function for parsing form values using
// gorilla/schema. It ensure that the Form is parsed (idempotent) and it will
// use a shared schema decoder.
func ParseReq(r *http.Request, reqStruct interface{}) error {
	if err := r.ParseForm(); err != nil {
		return err
	}

	if err := schemaDecoder.Decode(reqStruct, r.Form); err != nil {
		return err
	}

	return nil
}

// GetContext will return the context for a http Request.
func GetContext(w http.ResponseWriter, r *http.Request) context.Context {
	// Chitin panics if the ResponseWriter it gets is not its ResponseWriter.
	// This means that we need to unwrap our custom ResponseWriter before
	// calling chitin.Context.
	if customW, ok := w.(*ResponseWriter); ok {
		w = customW.ResponseWriter
	}

	ctx := chitin.Context(w, r)

	return context.WithValue(ctx, "GithubAccessToken", r.Header.Get("GithubAccessToken"))
}

type RouteOptions struct {
	AddCORS bool
}

func CreateHandler(mux *mux.Router, route string, f http.HandlerFunc, options *RouteOptions) *mux.Route {
	if options == nil {
		options = &RouteOptions{
			AddCORS: false,
		}
	}
	if options.AddCORS {
		mux.HandleFunc(route, AddCORSHeaders(f)).Methods("OPTIONS")
	}

	final := func(w http.ResponseWriter, r *http.Request) {
		// Capture panics, make sure we return a status code. In addition this
		// will cause them to be logged to rollbar.
		defer func() {
			if p := recover(); p != nil {
				JSONError(w,
					http.StatusInternalServerError,
					fmt.Sprintf("panic in %q", route),
					fmt.Errorf("Panic: %v", p), // Probably a better way to format this.
				)
				panic(p)
			}
		}()

		// // TODO: Need to find way to pass updated ctx to the child.
		// ctx := GetContext(w, r)
		// ctx = context.WithValue(ctx, "route", route)

		// Build a proxy response writer that will record the status. That way
		// we can use it when filing the statsd data.
		newWriter := &ResponseWriter{w, 200}

		graphiteName := strings.TrimLeft(route, "/")
		xact := timing.Xact{
			Stats: StatsdClient,
		}
		xact.AddName(strings.Join([]string{"endpoint", graphiteName}, "."))
		dinfo.IncreaseCounter(r.URL.Path)
		xact.Start()
		// This has to be wrapped in an anonymous function otherwise
		// newWriter.statusCode will be evaluated when the defer is
		// instantiated and not when final returns.
		defer func() {
			statusCode := strconv.Itoa(newWriter.statusCode)
			xact.End(statusCode)
			dinfo.IncreaseCounter(r.URL.Path + "-" + statusCode)
		}()

		f(newWriter, r)
	}

	return mux.Handle(route, http.HandlerFunc(final))
}
