package deployment

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strconv"
	"strings"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/dta/skadi/pkg/appconfig"
	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/freeze"
	githubttv "code.justin.tv/dta/skadi/pkg/github"
	"code.justin.tv/dta/skadi/pkg/helpers"
	"code.justin.tv/dta/skadi/pkg/info"
	pkgrepo "code.justin.tv/dta/skadi/pkg/repo"
	log "github.com/Sirupsen/logrus"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/jmoiron/sqlx"
)

const (
	ConsulPrefixDeployedVersion  = "deployed-version"
	ConsulPrefixKnownGoodVersion = "known-good-version"

	StatsNamePrefix = "deployment"
)

var (
	productionFlag         bool
	runtimeEnvironmentName string
	statsNamePrefix        string

	consulMasterDatacenters []string
	consulDatacenters       []string

	db             *DB
	consulClient   *consulapi.Client
	fsConsulClient *helpers.FSConsulClient
	workerID       string

	deploymentCache = make(map[int]Deployment)
	cinfo           = info.NewInfo("consul")
)

func RegisterHandlers(r *mux.Router, workerIdentifier string, dbInstance *sqlx.DB, cClient *consulapi.Client) {
	db = &DB{
		dbInstance,
	}
	consulClient = cClient
	fsConsulClient = helpers.NewFailSafeConsulClient(cClient)
	workerID = workerIdentifier

	config.CreateHandler(r, "/v1/deployments",
		CreateHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("POST")

	config.CreateHandler(r, "/v1/deployments/{id:[0-9]+}/status",
		GetStatusHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")

	config.CreateHandler(r, "/v1/deployments",
		ListHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")

	config.CreateHandler(r, "/v1/deployments/{id:[0-9]+}/log",
		GetLogHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")

	config.CreateHandler(r, "/v1/deployment/{id:[0-9]+}",
		GetDeploymentHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")

	config.CreateHandler(r, "/v1/codedeploy/{id:[0-9]+}",
		GetCodeDeployHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("POST")

	config.CreateHandler(r, "/v1/deployments/history",
		HistoryHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")

	appconfig.RegisterCallback(AppConfigCallback)
}

func SetRuntimeEnvironmentInfo(name string, production bool) {
	if name == "" {
		// let's make it sure
		panic("Empty runtime environment name is not possible")
	}

	runtimeEnvironmentName = name
	productionFlag = production

	statsNamePrefix = StatsNamePrefix

	if !productionFlag {
		statsNamePrefix += "_" + name
	}
}

func AppConfigCallback(c *appconfig.AppConfig) {
	log.Println("Reconfiguring deployment package")
	consulMasterDatacenters = helpers.UniqueStringsSorted(helpers.SplitStringNoEmpty(c.ConsulMasterDatacenters, ","))
	consulDatacenters = helpers.UniqueStringsSorted(helpers.SplitStringNoEmpty(c.ConsulDatacenters, ","))
}

type createDeploySchema struct {
	Branch        *string
	Ref           *string
	Environment   *string
	Owner         *string
	Repository    *string `schema:"repo"`
	Description   *string
	Creator       *string // (optional) username which a deploy is executed on behalf of.
	CodeReviewUrl string
	Severity      string
	TriggerSmoca  string

	// Targeted deploy related:
	// Targeted deploy can be used in 2 ways, a separate deploy job and linked deploy job
	// a separate deploy job can be used to deploy specific version to specific target hosts.
	// linked deploy is basically same but it creates a link to the parent deploy, so the deploy result can be
	// also tracked in a single(parent) deploy job
	Hosts *string // (optional) used for targeted deploy
	Link  *int64  // (optional) when used this deploy will be linked with given deploy id, used for redeploy.
}

type updateCodeDeploySchema struct {
	CodeDeployID string
}

func CreateHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)

	// Set the request with default values:
	deployReq := &createDeploySchema{}
	if err := config.ParseReq(r, deployReq); err != nil {
		config.JSONError(w, http.StatusBadRequest, "", err)
		return
	}

	for _, v := range []struct {
		name string
		val  *string
	}{
		{"branch", deployReq.Branch},
		{"environment", deployReq.Environment},
		{"owner", deployReq.Owner},
		{"repo", deployReq.Repository},
	} {
		if v.val == nil {
			err := fmt.Errorf("paramater %q is not set and is required", v.name)
			config.JSONError(w, http.StatusBadRequest, err.Error(), err)
			return
		}
	}

	// If we don't get a ref we can safely substitute the branch name in its
	// place.
	if deployReq.Ref == nil {
		deployReq.Ref = deployReq.Branch
	}

	repo := &pkgrepo.Repository{Owner: *deployReq.Owner, Name: *deployReq.Repository}

	githubClient, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "issue fetching github client", err)
		return
	}

	// Validate creator login if specified
	if deployReq.Creator != nil && *deployReq.Creator != "" {
		if _, _, err := githubClient.Users.Get(ctx, *deployReq.Creator); err != nil {
			config.JSONError(w, http.StatusBadRequest, "invalid creator login, "+*deployReq.Creator, err)
			return
		}
	}

	settings, err := pkgrepo.LoadSettings(githubClient, consulClient, *deployReq.Owner, *deployReq.Repository, *deployReq.Ref)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error loading settings", err)
		return
	}
	_, err = settings.Deploy.IsBranchWhiteListed(*deployReq.Environment, *deployReq.Branch)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error getting branch whitelist status", err)
		return
	}

	isFrozen, err := freeze.IsFrozen(*deployReq.Owner, *deployReq.Repository, *deployReq.Environment)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error getting frozen status", err)
		return
	}
	if isFrozen {
		config.JSONError(w, http.StatusInternalServerError, "can't deploy on frozen environment", nil)
		return
	}

	ghDeployment, err := CreateGithubDeployment(githubClient, repo, deployReq.Ref, deployReq.Environment, deployReq.Description, deployReq.CodeReviewUrl, deployReq.Severity, deployReq.TriggerSmoca, consulClient)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error creating github deployment", err)
		return
	}

	d, err := CreateDeployment(db, *deployReq.Branch, repo, deployReq.Creator, deployReq.Hosts, deployReq.Link, ghDeployment)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error creating deployment", err)
		return
	}

	user, _, err := githubClient.Users.Get(ctx, *d.Creator)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error getting github user", err)
		return
	}
	d.GithubCreator = user

	json.NewEncoder(w).Encode(d)
}

func GetStatusHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	// Validate ID
	idStr := mux.Vars(r)["id"]
	id, err := strconv.Atoi(idStr)
	if err != nil {
		config.JSONError(w,
			http.StatusBadRequest,
			fmt.Sprintf("can't parse id: %q", idStr),
			err)
		return
	}

	githubClient, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "issue fetching github client", err)
		return
	}

	jenkinsClient, err := config.JenkinsClient(ctx)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "issue fetching jenkins client", err)
		return
	}

	status, err := GetStatus(githubClient, jenkinsClient, id)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error executing deployment_status", err)
		return
	}
	if status == nil {
		config.JSONError(w, http.StatusNotFound, fmt.Sprintf("could not find deployment %d", id), nil)
		return
	}

	json.NewEncoder(w).Encode(status)
}

func ListHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	options := &ListDeploymentsOptions{}

	if err := config.ParseReq(r, options); err != nil {
		config.JSONError(w, http.StatusBadRequest, "", err)
		return
	}

	githubClient, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "issue fetching github client", err)
		return
	}

	// Adjust search input
	if helpers.IsEmptyStringp(options.Owner) && !helpers.IsEmptyStringp(options.Repository) {
		*options.Repository = strings.ToLower(*options.Repository)
		*options.Repository = helpers.SQLCompliantString(*options.Repository)
		r := helpers.SplitStringTrimmed(*options.Repository, "/")
		if len(r) == 2 {
			options.Owner = &r[0]
			options.Repository = &r[1]
		}
	}
	if !helpers.IsEmptyStringp(options.Environment) {
		*options.Environment = strings.ToLower(*options.Environment)
		*options.Environment = helpers.SQLCompliantString(*options.Environment)
	}

	deployments, err := ListDeployments(db, githubClient, options)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error executing list_deployments", err)
		return
	}
	json.NewEncoder(w).Encode(deployments)
}

func GetLogHandler(w http.ResponseWriter, r *http.Request) {
	idStr := mux.Vars(r)["id"]
	id, err := strconv.Atoi(idStr)
	if err != nil {
		config.JSONError(w,
			http.StatusBadRequest,
			fmt.Sprintf("can't parse id: %q", idStr),
			err)
		return
	}

	d, err := db.GetDeployment(int64(id))
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error getting deployment", err)
		return
	}

	tx, err := db.Begin()
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error starting db transaction", err)
		return
	}
	defer tx.Commit()

	l, err := d.Logs(tx)
	if err != nil && err.Error() == "sql: no rows in result set" {
		l := Log{DeploymentID: int64(id), Output: "No logs in database yet"}
		json.NewEncoder(w).Encode(l)
		return
	} else if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error getting logs %s", err)
		return
	}
	json.NewEncoder(w).Encode(l)
}

type messagedError struct {
	Err     error
	Message string
}

func (e *messagedError) Error() string {
	return fmt.Sprintf("%s: %s", e.Message, e.Err)
}

func getDeploymentWithCreatorAndComparison(githubClient *githubttv.Client, id int) (*Deployment, error) {
	logger := log.WithFields(log.Fields{"function": "getDeploymentWithCreatorAndComparison", "id": id})
	if deployment, ok := deploymentCache[id]; ok {
		logger.Debug("CACHE HIT")
		return &deployment, nil
	}
	logger.Debug("CACHE MISS")
	deployment, err := db.GetDeployment(int64(id))
	if err != nil {
		return nil, &messagedError{Message: "error getting deployment from db", Err: err}
	}
	deployCreator, err := deployment.GetCreator(githubClient)
	if err != nil {
		logger.Warnln("Failed to lookup creator. Trying GitHubCreator instead.", err.Error())
		deployCreator, err = deployment.GetGithubCreator(githubClient)
		if err != nil {
			return nil, &messagedError{Message: "error getting deployment github creator", Err: err}
		}
	}
	deployment.GithubCreator = deployCreator
	commitsComparison, err := deployment.GetCommitsComparison(githubClient)
	if err != nil {
		return nil, &messagedError{Message: "error getting deployment commits comparison", Err: err}
	}
	deployment.CommitsComparison = commitsComparison
	cacheDeployment(deployment, id)
	return deployment, nil
}

// Cache a deployment object for reuse, but only if the deployment failed
// or succeeded. This allows skadi to refresh the status if the deployment
// is in a pending or weird (unknown) state.
func cacheDeployment(deployment *Deployment, id int) bool {
	if *deployment.State == "failure" || *deployment.State == "success" {
		deploymentCache[id] = *deployment
		return true
	}
	return false
}

func getGithubClientFromContext(ctx context.Context) (*githubttv.Client, error) {
	c, err := config.GithubClient(ctx, false)
	if err != nil {
		return nil, &messagedError{Message: "error initializing github client", Err: err}
	}
	return &githubttv.Client{Client: c}, nil
}

func getIdFromRequest(r *http.Request) (int, error) {
	idStr := mux.Vars(r)["id"]
	id, err := strconv.Atoi(idStr)
	if err != nil {
		return 0, &messagedError{Message: fmt.Sprintf("can't parse id: %q", idStr), Err: err}
	}
	return id, nil
}

// GetDeploymentHandler responds with a deployment object if successful or an
// error object if not. In the request body, it accepts the parameter id int.
func GetDeploymentHandler(w http.ResponseWriter, r *http.Request) {
	var deployment *Deployment
	var githubClient *githubttv.Client

	id, err := getIdFromRequest(r)
	if err == nil {
		githubClient, err = getGithubClientFromContext(config.GetContext(w, r))
	}

	if err == nil {
		deployment, err = getDeploymentWithCreatorAndComparison(githubClient, id)
	}
	if err != nil {
		if merr, ok := err.(*messagedError); ok {
			config.JSONError(w, http.StatusInternalServerError, merr.Message, merr.Err)
		} else {
			config.JSONError(w, http.StatusInternalServerError, err.Error(), err)
		}
	}
	json.NewEncoder(w).Encode(deployment)
}

// GetCodeDeployHandler responds with a 200 deployment object if successful or an
// error object if not. In the request body, it accepts the parameter id int.
func GetCodeDeployHandler(w http.ResponseWriter, r *http.Request) {
	idStr := mux.Vars(r)["id"]
	id, err := strconv.Atoi(idStr)
	if err != nil {
		config.JSONError(w,
			http.StatusBadRequest,
			fmt.Sprintf("can't parse id: %q", idStr),
			err)
		return
	}
	codeDeploySchema := &updateCodeDeploySchema{}
	jsonContent, err := ioutil.ReadAll(r.Body)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error reading codedeploy request body", err)
		return
	}
	if err = json.Unmarshal(jsonContent, codeDeploySchema); err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error unmarshalling codedeploy json", err)
		return
	}
	err = db.UpdateDeploymentWithCodeDeployId(int64(id), codeDeploySchema.CodeDeployID)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error updating deployment with codedeploy id", err)
		return
	}
}

func HistoryHandler(w http.ResponseWriter, r *http.Request) {
	var oldestDeployTime *time.Time
	ctx := config.GetContext(w, r)
	options := &ListDeploymentsOptions{}

	if err := config.ParseReq(r, options); err != nil {
		config.JSONError(w, http.StatusBadRequest, "", err)
		return
	}

	githubClient, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "issue fetching github client", err)
		return
	}

	deploymentHistoryEvents := DeploymentHistoryEvents{}
	deployments, err := ListDeployments(db, githubClient, options)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error listing deployments", err)
		return
	}

	if len(deployments) >= 1 {
		oldestDeployTime = deployments[len(deployments)-1].CreatedAt
	} else {
		currentTime := time.Now()
		oldestDeployTime = &currentTime
	}

	for _, deploy := range deployments {
		dhe := DeploymentHistoryEvent{EventType: "deployment", EventDate: deploy.UpdatedAt, Deployment: deploy}
		deploymentHistoryEvents = append(deploymentHistoryEvents, &dhe)
	}

	freezeOptions := &freeze.ListFreezeOptions{
		Owner:       options.Owner,
		Repository:  options.Repository,
		Environment: options.Environment,
		AfterDate:   oldestDeployTime,
	}

	freezes, err := freeze.GetFreezes(freezeOptions)

	for _, freeze := range freezes {
		f := DeploymentHistoryEvent{EventType: "freeze", EventDate: freeze.CreatedAt, Freeze: freeze}
		deploymentHistoryEvents = append(deploymentHistoryEvents, &f)
	}

	json.NewEncoder(w).Encode(deploymentHistoryEvents)
}
