package deployment

import (
	"bytes"
	"fmt"
	"log"
	"strings"

	"github.com/google/go-github/github"
	"github.com/jmoiron/sqlx"
	"github.com/lann/squirrel"
)

type DB struct {
	*sqlx.DB
}

type Tx struct {
	*sqlx.Tx
}

type Log struct {
	DeploymentID int64  `json:"deployment_id" db:"deployment_id"`
	Output       string `json:"output" db:"output"`
}

func (db *DB) Begin() (*Tx, error) {
	tx, err := db.Beginx()
	if err != nil {
		return nil, err
	}

	return &Tx{tx}, nil
}

func GetDB() *DB {
	return db
}

/****************************************************************************
** TABLE deployments related
*****************************************************************************/

func (tx *Tx) InsertDeployment(d *Deployment) error {
	rows, err := tx.NamedQuery(`INSERT INTO deployments (
		githubid, owner, repository, sha, branch, environment, creator, state, description, previoussha,
		codereviewurl, severity, hosts, link
	) VALUES (
		:githubid, :owner, :repository, :sha, :branch, :environment, :creator, :state, :description, :previoussha,
		:codereviewurl, :severity, :hosts, :link
	) RETURNING ID`,
		d,
	)
	if err != nil {
		return err
	}
	defer rows.Close()

	for rows.Next() {
		if err = rows.Scan(&d.ID); err != nil {
			return err
		}
		break
	}

	return nil
}

// UpdateDeploymentState updates on deployments table.
func (db *DB) UpdateDeploymentState(githubID int64, state string) error {
	result, err := db.Exec("UPDATE deployments set state = $1, updatedat = now() where githubid = $2", state, githubID)
	if err != nil {
		return err
	}
	num, err := result.RowsAffected()
	if num != 1 {
		return fmt.Errorf("expected one row to be updated, instead %v", num)
	}
	return nil
}

// GetDeployment fetches and returns a Deployment from the database.
func (db *DB) GetDeployment(id int64) (*Deployment, error) {
	var d Deployment
	err := db.Get(&d, "SELECT * FROM deployments WHERE id=$1", id)
	if err != nil {
		return nil, err
	}
	return &d, nil
}

func (db *DB) FindDeploymentByGithubID(githubID int64) (*Deployment, error) {
	var d Deployment
	err := db.Get(&d, "SELECT * FROM deployments WHERE githubid=$1", githubID)
	if err != nil {
		return nil, err
	}
	return &d, nil
}

func (db *DB) ListDeployments(options *ListDeploymentsOptions) ([]*Deployment, error) {
	// Set the request with default values:
	if options.Count == 0 {
		options.Count = 10
	}

	psql := squirrel.StatementBuilder.PlaceholderFormat(squirrel.Dollar)

	// hits a database which we can filter based on time, or repo/environment
	results := []*Deployment{}

	// Build basic query:
	query := psql.Select("*").From("deployments")

	// Dynamically build the rest of the query.
	for k, v := range map[string]*string{
		"owner":       options.Owner,
		"creator":     options.Creator,
		"repository":  options.Repository,
		"branch":      options.Branch,
		"environment": options.Environment,
		"state":       options.State,
		"sha":         options.SHA,
		"previoussha": options.PreviousSHA,
	} {
		if v != nil {
			if v2 := strings.TrimSpace(*v); v2 != "" {
				comp := "="
				if strings.Contains(v2, "%") {
					comp = "LIKE"
				}
				query = query.Where(fmt.Sprintf("%s %s ?", k, comp), v2)
			}
		}
	}
	if options.Start != nil && *options.Start != "" {
		query = query.Where("createdat >= ?", *options.Start)
	}
	if options.End != nil && *options.End != "" {
		query = query.Where("createdat <= ?", *options.End)
	}

	query = query.OrderBy("createdat desc").Limit(options.Count)
	q, args, err := query.ToSql()
	if err != nil {
		return nil, err
	}

	rows, err := db.Queryx(q, args...)
	if err != nil {
		return nil, fmt.Errorf("Error querying db: %v", err)
	}
	defer rows.Close()

	for rows.Next() {
		var deploy Deployment

		err := rows.StructScan(&deploy)
		if err != nil {
			return nil, err
		}

		results = append(results, &deploy)
	}

	return results, nil
}

func (db *DB) DeleteDeployment(id int64) error {
	_, err := db.Exec("DELETE FROM deployments where id=$1", id)
	return err
}

// NOTE:we're using githubID because it's the unique identifier sent back from GHE as part of the Deployment object.
func (db *DB) UpdateDeploymentWithJenkinsParams(githubId int64, jenkinsJob string, hosts string) error {
	_, err := db.Exec("UPDATE deployments set jenkinsjob = $1, hosts = $2 where githubId = $3", jenkinsJob, hosts, githubId)
	return err
}

func (db *DB) UpdateDeploymentWithCodeDeployId(id int64, codeDeployID string) error {
	codeDeployID = strings.TrimSpace(codeDeployID)
	_, err := db.Exec("UPDATE deployments set codedeployid = $1 where id = $2", codeDeployID, id)
	return err
}

/****************************************************************************
** TABLE deployment_log related
*****************************************************************************/

// db.AppendLog is a convenient helper method for tx.AppendLog. `id` is deployment_id not git_id
func (db *DB) AppendLog(text string, id int64) error {
	tx, err := db.Begin()
	if err != nil {
		return err
	}
	defer tx.Commit()
	return tx.AppendLog(text, id)
}

// db.AppendLogGithub is a convenient helper method for tx.AppendLogGithub
func (db *DB) AppendLogGithub(text string, deploy *github.Deployment, repo *github.Repository) error {
	tx, err := db.Begin()
	if err != nil {
		return err
	}
	defer tx.Commit()
	return tx.AppendLogGithub(text, deploy, repo)
}

func (tx *Tx) GetLog(id int64) (*Log, error) {
	var l Log
	err := tx.Get(&l, "SELECT * FROM deployment_logs WHERE deployment_id=$1", id)
	if err != nil {
		return nil, err
	}

	return &l, nil
}

func (tx *Tx) AppendLog(text string, id int64) error {
	var count int
	err := tx.Get(&count, "SELECT count(*) FROM deployment_logs WHERE deployment_id=$1", id)
	if err != nil {
		return err
	}

	if count == 0 {
		rows, err := tx.Queryx("INSERT INTO deployment_logs (deployment_id) VALUES ($1)", id)
		if err != nil {
			return err
		}
		rows.Close()
	}

	var buf bytes.Buffer
	lines := strings.Split(text, "\n")
	logger := log.New(&buf, "", log.LstdFlags)
	for _, l := range lines {
		logger.Print(l)
	}

	rows, err := tx.Queryx("UPDATE deployment_logs SET output = COALESCE(output, '') || $1 WHERE deployment_id=$2", buf.String(), id)
	if err != nil {
		return err
	}
	rows.Close()

	return nil
}

func (tx *Tx) AppendLogGithub(text string, deploy *github.Deployment, repo *github.Repository) error {
	var id int64
	err := tx.Get(&id, "SELECT id from deployments WHERE githubid = $1 AND owner = $2 AND repository = $3", *deploy.ID, *repo.Owner.Login, *repo.Name)
	if err != nil {
		return err
	}

	return tx.AppendLog(text, id)
}

// AppendLogGithubForEvent appends a message to git from event structure.
// This method is designed to be called outside the deploy object with event structure.
func AppendLogGithubForEvent(text string, event *Event) error {
	if db == nil {
		return fmt.Errorf("db instance hasn't set.")
	}
	tx, err := db.Begin()
	if err != nil {
		return err
	}
	defer tx.Commit()
	tx.AppendLogGithub(text, event.Deployment, event.Repository)
	return nil
}

/****************************************************************************
** TABLE deployment_status related
*****************************************************************************/
func (db *DB) CreateDeploymentStatus(target string, state string, deployer string) error {
	_, err := db.Exec("INSERT INTO deployment_status (target, worker, state, deployer) VALUES ($1, $2, $3, $4)", target, workerID, state, deployer)
	return err
}

func (db *DB) UpdateMyDeploymentStatus(target string, state string, deployer string) error {
	result, err := db.Exec("UPDATE deployment_status SET state = $1, deployer = $2, updatedat = now() WHERE target = $3 AND worker = $4", state, deployer, target, workerID)
	if err != nil {
		return err
	}
	num, err := result.RowsAffected()
	if num != 1 {
		return fmt.Errorf("Failed to update deployment status. Target:%v, State:%v", target, state)
	}
	return nil
}

func (db *DB) DeleteMyDeploymentStatus(target string) error {
	_, err := db.Exec("DELETE FROM deployment_status WHERE target = $1 AND worker = $2", target, workerID)
	return err
}

func (db *DB) ListMyPendingJobs() ([]DeploymentStatus, error) {
	s := []DeploymentStatus{}
	err := db.Select(&s, "SELECT * FROM deployment_status WHERE worker = $1", workerID)
	return s, err
}

func (db *DB) IsTargetRunningByOthers(target string) (bool, error) {
	s := []DeploymentStatus{}
	err := db.Select(&s, "SELECT * FROM deployment_status WHERE target = $1 AND worker != $2", target, workerID)
	if err != nil {
		return true, err
	}
	if len(s) > 0 {
		return true, nil
	}
	return false, nil
}

func (db *DB) AutoDeployList() (*[]AutoDeploy, error) {
	autodeploy := []AutoDeploy{}
	err := db.Select(&autodeploy, "SELECT ID, owner, repository, environment, reason, autodeploy, enddate FROM freezeschedules WHERE autodeploy=TRUE AND enddate < current_timestamp AND autodeploytime IS NULL;")
	if err != nil {
		return nil, err
	}
	return &autodeploy, nil
}

func (db *DB) AutoDeploySent(id int) error {
	result, err := db.Exec("UPDATE freezeschedules SET autodeploytime=now() WHERE ID=$1", id)
	if err != nil {
		return err
	}
	num, err := result.RowsAffected()
	if num != 1 {
		return fmt.Errorf("expected one row to be updated, instead %v", num)
	}
	return nil
}

func (db *DB) GetLastMasterDeploySha(owner, repository, branch string) (string, error) {
	result := ""
	row, err := db.Query("SELECT sha FROM deployments WHERE owner=$1 AND repository=$2 AND branch=$3 ORDER BY createdat DESC LIMIT 1", owner, repository, branch)
	if err != nil {
		return "", err
	}
	for row.Next() {
		err = row.Scan(&result)
		if err != nil {
			return "", err
		}
	}

	return result, nil

}
