package deployment

import (
	"math/rand"
	"regexp"
	"testing"

	"github.com/google/go-github/github"
	"github.com/jmoiron/sqlx"
	. "github.com/smartystreets/goconvey/convey"
)

func TestAppendLog(t *testing.T) {
	pg, err := sqlx.Open("postgres", "sslmode=disable")
	if err != nil {
		t.Fatal(err)
	}
	db := &DB{pg}

	tx, err := db.Begin()
	if err != nil {
		t.Fatal(err)
	}
	defer tx.Rollback()

	i := int64(rand.Int31())
	d := &Deployment{
		GithubID: &i,
	}
	err = tx.InsertDeployment(d)
	if err != nil {
		t.Fatal(err)
	}

	err = tx.AppendLog("foo", *d.ID)
	if err != nil {
		t.Fatal(err)
	}

	err = tx.AppendLog("bar", *d.ID)
	if err != nil {
		t.Fatal(err)
	}

	l, err := d.Logs(tx)
	if err != nil {
		t.Fatal(err)
	}

	regex := regexp.MustCompile("\\d*/\\d*/\\d* \\d*:\\d*:\\d* foo\n\\d*/\\d*/\\d* \\d*:\\d*:\\d* bar\n")

	if !regex.MatchString(l.Output) {
		t.Fatalf("unexpected Description: %v", l.Output)
	}
}

func TestAppendLogGithub(t *testing.T) {
	Convey("with a database connection", t, func(c C) {
		pg, err := sqlx.Open("postgres", "sslmode=disable")
		c.So(err, ShouldBeNil)

		db := &DB{pg}
		tx, err := db.Begin()
		c.So(err, ShouldBeNil)
		defer tx.Rollback()

		c.Convey("and a skadi deployment", func(c C) {
			owner := "release"
			name := "example"
			i := int64(rand.Int31())
			d := &Deployment{
				GithubID:   &i,
				Owner:      &owner,
				Repository: &name,
			}
			err = tx.InsertDeployment(d)
			c.So(err, ShouldBeNil)

			c.Convey("AppendLogGithub writes to the deploy log using github structs", func(c C) {
				gh := &github.Deployment{
					ID: &i,
				}

				repo := &github.Repository{
					Owner: &github.User{Login: &owner},
					Name:  &name,
				}

				err := tx.AppendLogGithub("foo", gh, repo)
				c.So(err, ShouldBeNil)
			})
		})
	})
}
