package deployment

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/freeze"
	githubttv "code.justin.tv/dta/skadi/pkg/github"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"code.justin.tv/dta/skadi/pkg/helpers"
	"code.justin.tv/dta/skadi/pkg/repo"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
)

// Event contains information passed from the github webhook after a deployment is created
type Event struct {
	Deployment *github.Deployment
	Repository *github.Repository
	Sender     *github.User
}

// Deployment represents a rollout of an application change
type Deployment struct {
	ID            *int64  `json:"id,omitempty" db:"id"`
	GithubID      *int64  `json:"github_id,omitempty" db:"githubid"`
	CodeDeployID  *[]byte `json:"codedeploy_id,omitempty" db:"codedeployid"`
	Repository    *string `json:"repository,omitempty" db:"repository"`
	Owner         *string `json:"owner,omitempty" db:"owner"`
	SHA           *string `json:"sha,omitempty" db:"sha"`
	Branch        *string `json:"branch,omitempty" db:"branch"`
	Environment   *string `json:"environment,omitempty" db:"environment"`
	Creator       *string `json:"creator,omitempty" db:"creator"`
	State         *string `json:"state,omitempty" db:"state"`
	Description   *string `json:"description,omitempty" db:"description"`
	PreviousSHA   *string `json:"previous_sha,omitempty" db:"previoussha"`
	CodeReviewURL *string `json:"code_review_url,omitempty" db:"codereviewurl"`
	Severity      *string `json:"severity,omitempty" db:"severity"`
	Hosts         *string `json:"hosts,omitempty" db:"hosts"`
	JenkinsJob    *string `json:"jenkinsjob,omitempty" db:"jenkinsjob"`
	Link          *int64  `json:"link,omitempty" db:"link"`

	GithubCreator     *github.User              `json:"github_creator"`
	CommitsComparison *github.CommitsComparison `json:"commits_comparison"`

	CreatedAt *time.Time `json:"created_at,omitempty" db:"createdat"`
	UpdatedAt *time.Time `json:"updated_at,omitempty" db:"updatedat"`
}

func (d *Deployment) ToString(str *string) string {
	if str != nil {
		return *str
	}
	return ""
}

func (d *Deployment) ToInt64(i *int64) int64 {
	if i != nil {
		return *i
	}
	return 0
}
func (d *Deployment) UnixTime(t *time.Time) int64 {
	if t != nil {
		return t.Unix()
	}
	return 0
}

// DeploymentHistory is a container that holds events related to deployments
type DeploymentHistoryEvent struct {
	EventType  string         `json:"event_type"`
	EventDate  *time.Time     `json:"event_date"`
	Deployment *Deployment    `json:"deployment,omitempty"`
	Freeze     *freeze.Freeze `json:"freeze"`
}

type DeploymentHistoryEvents []*DeploymentHistoryEvent

func (slice DeploymentHistoryEvents) Len() int {
	return len(slice)
}

func (slice DeploymentHistoryEvents) Less(i, j int) bool {
	if slice[i].EventDate != nil && slice[j].EventDate != nil {
		return slice[i].EventDate.After(*slice[j].EventDate)
	}
	return false
}

func (slice DeploymentHistoryEvents) Swap(i, j int) {
	slice[i], slice[j] = slice[j], slice[i]
}

func (d *Deployment) Logs(tx *Tx) (*Log, error) {
	return tx.GetLog(*d.ID)
}

// DeploymentStatus represents a row from deployment_status table.
type DeploymentStatus struct {
	Target    *string
	Worker    *string
	State     *string
	Deployer  *string
	CreatedAt *time.Time
	UpdatedAt *time.Time
}

type SkadiDeploymentPayload struct {
	PreviousSHA   string `json:"previous_sha"`
	CodeReviewURL string `json:"code_review_url,omitempty"`
	Severity      string `json:"severity,omitempty"`
	TriggerSmoca  string `json:"trigger_smoca,omitempty"`
}

func ParsePayload(p json.RawMessage) (*SkadiDeploymentPayload, error) {
	var payload SkadiDeploymentPayload
	s, err := strconv.Unquote(string(p))
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal([]byte(s), &payload)
	if err != nil {
		return nil, err
	}

	return &payload, nil
}

// NewDeployment will take a branch, the repo info and the deployment info and
// prepare a deployment response. The deployment object can be serialized for
// user requests or to input into the database.
func NewDeployment(branch *string, repo *repo.Repository, hosts *string, link *int64, githubDeployment *github.Deployment) (*Deployment, error) {
	if branch == nil {
		return nil, fmt.Errorf("invalid branch")
	}
	if repo == nil {
		return nil, fmt.Errorf("repo is nil")
	}
	if githubDeployment == nil {
		return nil, fmt.Errorf("githubDeployment is nil")
	}
	if githubDeployment.Creator == nil || githubDeployment.CreatedAt == nil {
		return nil, fmt.Errorf("invalid github deployment")
	}

	// Refactor hosts variable - split, unique then join without empty string
	if hosts != nil {
		if hosts2 := helpers.JoinStringsNoEmpty(helpers.UniqueStringsSorted(helpers.SplitStringNoEmpty(*hosts, ",")), ","); hosts2 == "" {
			hosts = nil
		} else {
			hosts = &hosts2
		}
	}

	payload, err := ParsePayload(githubDeployment.Payload)
	if err != nil {
		return nil, fmt.Errorf("ParsePayload returned err in NewDeployment: %v", err)
	}

	d := &Deployment{
		GithubID:      githubDeployment.ID,
		Repository:    &repo.Name,
		Owner:         &repo.Owner,
		SHA:           githubDeployment.SHA,
		Branch:        branch,
		Environment:   githubDeployment.Environment,
		CreatedAt:     &githubDeployment.CreatedAt.Time,
		Creator:       githubDeployment.Creator.Login,
		Description:   githubDeployment.Description,
		PreviousSHA:   &payload.PreviousSHA,
		CodeReviewURL: &payload.CodeReviewURL,
		Severity:      &payload.Severity,
		Hosts:         hosts,
		Link:          link,
	}

	if githubDeployment.UpdatedAt != nil {
		d.UpdatedAt = &githubDeployment.UpdatedAt.Time
	}

	return d, nil
}

// FindDeployment will take a github deployment ID, find the corresponding
// skadi deployment in the database, and return it.
func FindDeployment(githubID int64) (*Deployment, error) {
	return db.FindDeploymentByGithubID(githubID)
}

func CreateGithubDeployment(client *github.Client, r *repo.Repository, ref, environment, description *string, codeReviewURL, severity, triggerSmoca string, consul *consulapi.Client) (*github.Deployment, error) {
	// always read config from master when looking up build_status_context
	deployConfig, err := config.LoadDeployConfig(client, consul, r.Owner, r.Name, githubcache.GetDefaultBranch(r.Owner, r.Name))
	if err != nil {
		return nil, err
	}

	sha, err := previousDeployedSHA(r.FullName(), *environment, consul.KV())
	if err != nil {
		log.Printf("could not read previously deployed sha: %v", err)
	}

	payload := SkadiDeploymentPayload{
		PreviousSHA:   sha,
		CodeReviewURL: codeReviewURL,
		Severity:      severity,
		TriggerSmoca:  triggerSmoca,
	}
	b, err := json.Marshal(payload)
	if err != nil {
		return nil, err
	}
	payloadJson := string(b)

	autoMerge := false
	deployRequest := &github.DeploymentRequest{
		Ref:              ref,
		Environment:      environment,
		AutoMerge:        &autoMerge,
		Description:      description,
		Payload:          &payloadJson,
		RequiredContexts: deployConfig.EnvironmentRequiredContexts(*environment),
	}

	deployment, _, err := client.Repositories.CreateDeployment(context.TODO(), r.Owner, r.Name, deployRequest)
	if err != nil {
		return nil, err
	}

	return deployment, nil
}

func CreateDeployment(db *DB, branch string, repo *repo.Repository, creator *string, hosts *string, link *int64, githubDeployment *github.Deployment) (*Deployment, error) {
	d, err := NewDeployment(&branch, repo, hosts, link, githubDeployment)
	if err != nil {
		return nil, err
	}

	tx, err := db.Begin()
	if err != nil {
		return nil, err
	}
	defer tx.Commit()

	// Replace creator if set and update description.
	if creator != nil && len(*creator) > 0 && *creator != *d.Creator {
		*d.Description += " - by " + *d.Creator + " on behalf of " + *creator
		d.Creator = creator
	}

	state := StateUnknown
	d.State = &state
	err = tx.InsertDeployment(d)
	if err != nil {
		return nil, err
	}

	log.Printf("new deployment created. DeployID=%v, GitID=%v", *d.ID, *d.GithubID)
	return d, nil
}

func GetCurrentDeploymentForEnviroment(db *DB, r *repo.Repository, client *github.Client, environment string) (*Deployment, error) {
	state := StateSuccess
	deploys, err := db.ListDeployments(&ListDeploymentsOptions{
		Count:       1,
		State:       &state,
		Environment: &environment,
		Owner:       &r.Owner,
		Repository:  &r.Name,
	})
	if err != nil {
		return nil, err
	}

	if len(deploys) == 0 {
		return nil, fmt.Errorf("no successful deploy found for %v", environment)
	}

	deploy := deploys[0]
	log.WithFields(log.Fields{"func": "GetCurrentDeploymentForEnvironment"}).Debugf("deploy=%v", deploy)

	user, _, err := client.Users.Get(context.TODO(), *deploy.Creator)
	if err != nil {
		return nil, err
	}
	deploy.GithubCreator = user

	return deploy, nil
}

type ListDeploymentsOptions struct {
	Owner       *string
	Creator     *string
	Repository  *string `schema:"repo"`
	Branch      *string
	Environment *string
	State       *string
	SHA         *string
	PreviousSHA *string
	Count       uint64
	Start       *string
	End         *string
}

func parseGithubCreatorIfNameChange(d *Deployment, client *github.Client) (*github.User, error) {
	opt := &github.DeploymentsListOptions{SHA: *d.SHA}
	deployments, _, err := client.Repositories.ListDeployments(context.TODO(), *d.Owner, *d.Repository, opt)
	if err != nil {
		return nil, err
	}
	for _, i := range deployments {
		if *i.ID == *d.GithubID {
			// The deployment struct might have enough info of the creator that we don't have to call the users api
			user, _, err := client.Users.Get(context.TODO(), *i.Creator.Login)
			if err != nil {
				return nil, err
			}
			return user, nil
		}
	}
	return nil, errors.New("No user found")
}

func ListDeployments(db *DB, client *github.Client, options *ListDeploymentsOptions) ([]*Deployment, error) {
	results, err := db.ListDeployments(options)
	if err != nil {
		return nil, err
	}

	for _, d := range results {
		user, _, err := client.Users.Get(context.TODO(), *d.Creator)
		if err != nil {
			user, err = parseGithubCreatorIfNameChange(d, client)
			if err != nil {
				return nil, err
			}
		}
		d.GithubCreator = user
	}

	return results, nil
}

type kvGetter interface {
	Get(string, *consulapi.QueryOptions) (*consulapi.KVPair, *consulapi.QueryMeta, error)
}

func previousDeployedSHA(fullName, env string, getter kvGetter) (string, error) {
	kv, _, err := getter.Get(fmt.Sprintf("%v/%v/%v", ConsulPrefixKnownGoodVersion, fullName, env), nil)
	if err != nil {
		return "", err
	}
	if kv == nil {
		return "", nil
	}

	str := string(kv.Value)
	return str, nil
}

// GetGithubCreatorClient is a client interface that encapsulates all the
// functions the *Deployment.GetGithubCreator() method needs.
type GetGithubCreatorClient interface {
	githubttv.ListDeployments
	githubttv.GetUser
}

// GetCreator returns the deploy user info
func (d *Deployment) GetCreator(githubClient GetGithubCreatorClient) (*github.User, error) {
	if d.Creator == nil {
		return nil, errors.New("deployment creator is nil")
	}
	user, err := githubClient.GetUser(*d.Creator)
	if err != nil {
		return nil, err
	}
	return user, nil
}

// GetGithubCreator returns the actual github User that created the deployment.
func (d *Deployment) GetGithubCreator(githubClient GetGithubCreatorClient) (*github.User, error) {
	if d.GithubID == nil {
		return nil, errors.New("deployment github id is nil")
	}
	if d.Owner == nil {
		return nil, errors.New("deployment owner is nil")
	}
	if d.Repository == nil {
		return nil, errors.New("deployment repository is nil")
	}
	if d.SHA == nil {
		return nil, errors.New("deployment sha is nil")
	}
	if d.Environment == nil {
		return nil, errors.New("deployment environment is nil")
	}
	deployments, err := githubClient.ListDeployments(*d.Owner, *d.Repository, &github.DeploymentsListOptions{SHA: *d.SHA, Environment: *d.Environment})
	if err != nil {
		return nil, err
	}
	for _, deployment := range deployments {
		if deployment.ID == nil {
			continue
		}
		if *deployment.ID == *d.GithubID {
			if deployment.Creator == nil {
				return nil, errors.New("found deployment, but creator is nil")
			}
			return deployment.Creator, nil
		}
	}
	return nil, errors.New("could not find deployment creator")
}

// GetCommitsComparisonClient is a client interface that encapsulates all the
// functions the *Deployment.GetCommitsComparison() method needs.
type GetCommitsComparisonClient interface {
	githubttv.CompareRepositoryCommits
}

// GetCommitsComparison returns a CommitsComparison between deployment.SHA and
// deployment.PreviousSHA. If PreviousSHA is empty, it compares deployment.SHA
// against itself.
func (d *Deployment) GetCommitsComparison(githubClient GetCommitsComparisonClient) (*github.CommitsComparison, error) {
	if d.PreviousSHA == nil || strings.TrimSpace(*d.PreviousSHA) == "" {
		d.PreviousSHA = d.SHA
	}
	if d.SHA == nil {
		return nil, errors.New("deployment SHA is nil")
	}
	if d.Owner == nil {
		return nil, errors.New("deployment owner is nil")
	}
	if d.Repository == nil {
		return nil, errors.New("deployment repository is nil")
	}
	commitsComparison, err := githubClient.CompareRepositoryCommits(*d.Owner, *d.Repository, *d.PreviousSHA, *d.SHA)
	if err != nil {
		return nil, err
	}
	return commitsComparison, nil
}
