package deployment

import (
	"errors"
	"fmt"
	"math/rand"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"
	"time"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/dta/skadi/pkg/repo"

	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/jmoiron/sqlx"
	_ "github.com/lib/pq"
)

type stubKV struct{}

func (k *stubKV) Get(key string, q *consulapi.QueryOptions) (*consulapi.KVPair, *consulapi.QueryMeta, error) {
	return nil, nil, nil
}

func setupGithub() (*http.ServeMux, *httptest.Server, *github.Client) {
	mux := http.NewServeMux()
	server := httptest.NewServer(mux)

	client := github.NewClient(nil)
	url, _ := url.Parse(server.URL + "/")
	client.BaseURL = url

	return mux, server, client
}

func generateDeploymentEvent() *Event {
	name := "web/countess"
	env := "staging"
	sha := "5678"
	repo := github.Repository{FullName: &name}
	d := generateGithubDeployment(env, sha, "", "", "", "")
	return &Event{
		Deployment: d,
		Repository: &repo,
	}
}

func generateGithubDeployment(env, sha, desc, creator, review, severity string) *github.Deployment {
	id := int64(rand.Int31())
	payload := []byte(fmt.Sprintf("\"{\\\"previous_sha\\\":\\\"67212d5eb03550f34d9057e04a6c21880193321d\\\",\\\"code_review_url\\\":\\\"%s\\\",\\\"severity\\\":\\\"%s\\\"}\"", review, severity))
	return &github.Deployment{
		ID:          &id,
		Environment: &env,
		SHA:         &sha,
		Description: &desc,
		CreatedAt:   &github.Timestamp{Time: time.Now()},
		Creator:     &github.User{Login: &creator},
		Payload:     payload,
	}
}

func TestListDeploymentsEmbedUser(t *testing.T) {
	pg, err := sqlx.Open("postgres", "sslmode=disable")
	if err != nil {
		log.Fatal(err)
	}
	db := &DB{pg}

	mux, server, client := setupGithub()
	defer server.Close()

	mux.HandleFunc("/users/kthornhill", func(w http.ResponseWriter, r *http.Request) {
		fmt.Fprint(w, `{"id":1,"name":"Keith Thornhill"}`)
	})

	name := "test_repo"
	repo := &repo.Repository{Owner: "release", Name: name}
	gh := generateGithubDeployment("devel", "ABCD", "", "kthornhill", "", "")

	d, err := CreateDeployment(db, "master", repo, nil, nil, nil, gh)
	if err != nil {
		t.Fatal(err)
	}
	defer db.DeleteDeployment(*d.ID)

	deploys, err := ListDeployments(db, client, &ListDeploymentsOptions{Repository: &name})
	if err != nil {
		t.Error(err)
	}

	if len(deploys) != 1 {
		t.Errorf("expected one record, got %v", len(deploys))
	}

	if *deploys[0].GithubCreator.Name != "Keith Thornhill" {
		t.Errorf("unexpected creator name: %v", *deploys[0].GithubCreator.Name)
	}
}

func TestParsePayload(t *testing.T) {
	previousSha := "67212d5eb03550f34d9057e04a6c21880193321d"
	codeReviewURL := "http://git.xarth.tv/dta/skadi/pulls/1"
	severity := "low"
	str := fmt.Sprintf("\"{\\\"previous_sha\\\":\\\"%s\\\",\\\"code_review_url\\\":\\\"%s\\\",\\\"severity\\\":\\\"%s\\\"}\"",
		previousSha, codeReviewURL, severity)
	log.Print(str)
	payload, err := ParsePayload([]byte(str))
	if err != nil {
		t.Fatal(err)
	}

	if payload.PreviousSHA != previousSha {
		t.Fatalf("payload did not parse correctly: %+v", payload)
	}
	if payload.CodeReviewURL != codeReviewURL {
		t.Fatalf("payload did not parse correctly: %+v", payload)
	}
	if payload.Severity != severity {
		t.Fatalf("payload did not parse correctly: %+v", payload)
	}
}

func TestCreateDeploymentDescription(t *testing.T) {
	pg, err := sqlx.Open("postgres", "sslmode=disable")
	if err != nil {
		t.Fatal(err)
	}
	db := &DB{pg}

	repo := &repo.Repository{Owner: "release", Name: "example"}
	gh := generateGithubDeployment("production", "deadbeef", "foobar", "kthornhill", "", "")

	d1, err := CreateDeployment(db, "master", repo, nil, nil, nil, gh)
	if err != nil {
		t.Fatal(err)
	}
	defer db.DeleteDeployment(*d1.ID)

	d2, err := db.GetDeployment(*d1.ID)
	if err != nil {
		t.Fatal(err)
	}
	if *d2.Description != "foobar" {
		t.Fatal("deployment description not written to db during creation")
	}
}

func TestCreateDeploymentWithCreator(t *testing.T) {
	pg, err := sqlx.Open("postgres", "sslmode=disable")
	if err != nil {
		t.Fatal(err)
	}
	db := &DB{pg}
	defer db.Close()

	repo := &repo.Repository{Owner: "release", Name: "example"}
	gh := generateGithubDeployment("production", "deadbeef", "foobar", "kthornhill", "", "")
	creator := "seungyou"

	d1, err := CreateDeployment(db, "master", repo, &creator, nil, nil, gh)
	if err != nil {
		t.Fatal(err)
	}
	defer db.DeleteDeployment(*d1.ID)

	d2, err := db.GetDeployment(*d1.ID)
	if err != nil {
		t.Fatal(err)
	}
	if *d2.Creator != creator {
		t.Fatal("deployment creator is not matching.")
	}
	if d2.Hosts != nil {
		t.Fatal("hosts value should be nil.")
	}
	if d2.Link != nil {
		t.Fatal("link value should be nil.")
	}
}

func TestCreateDeploymentWithHostsAndLink(t *testing.T) {
	pg, err := sqlx.Open("postgres", "sslmode=disable")
	if err != nil {
		t.Fatal(err)
	}
	db := &DB{pg}
	defer db.Close()

	repo := &repo.Repository{Owner: "release", Name: "example"}
	gh := generateGithubDeployment("production", "deadbeef", "foobar", "kthornhill", "", "")
	creator := "seungyou"

	hosts := "testhost2.com, testhost1.com,  testhost3.com"
	link := int64(1234)
	d1, err := CreateDeployment(db, "master", repo, &creator, &hosts, &link, gh)
	if err != nil {
		t.Fatal(err)
	}
	defer db.DeleteDeployment(*d1.ID)

	d2, err := db.GetDeployment(*d1.ID)
	if err != nil {
		t.Fatal(err)
	}
	if d2.Hosts == nil || *d2.Hosts != "testhost1.com,testhost2.com,testhost3.com" {
		t.Fatal("hosts value is not matching.")
	}
	if d2.Link == nil || *d2.Link != link {
		t.Fatal("link value is not matching.")
	}
}

func TestGetCurrentDeploymentForEnviroment(t *testing.T) {
	mux, server, client := setupGithub()
	defer server.Close()

	mux.HandleFunc("/users/kthornhill", func(w http.ResponseWriter, r *http.Request) {
		fmt.Fprint(w, `{"id":1,"name":"Keith Thornhill"}`)
	})

	db, deploy1, repo := insertDeployment(t)
	defer db.DeleteDeployment(*deploy1.ID)
	db.UpdateDeploymentState(*deploy1.GithubID, "success")

	time.Sleep(2 * time.Second)

	db, deploy2, _ := insertDeployment(t)
	defer db.DeleteDeployment(*deploy2.ID)
	db.UpdateDeploymentState(*deploy2.GithubID, "failure")

	current, err := GetCurrentDeploymentForEnviroment(db, repo, client, "production")
	if err != nil {
		t.Fatal(err)
	}

	if *current.ID != *deploy1.ID {
		t.Fatalf("current lookup returned wrong deploy")
	}
}

func insertDeployment(t *testing.T) (*DB, *Deployment, *repo.Repository) {
	pg, err := sqlx.Open("postgres", "sslmode=disable")
	if err != nil {
		t.Fatal(err)
	}
	db := &DB{pg}

	repo := &repo.Repository{Owner: "release", Name: "example"}
	gh := generateGithubDeployment("production", "deadbeef", "", "kthornhill", "", "")

	d, err := CreateDeployment(db, "master", repo, nil, nil, nil, gh)
	if err != nil {
		t.Fatal(err)
	}

	return db, d, repo
}

type GetGithubCreatorTestClient struct {
	ID      *int64
	creator *github.User
}

func (c *GetGithubCreatorTestClient) ListDeployments(owner, repo string, opt *github.DeploymentsListOptions) ([]*github.Deployment, error) {
	return []*github.Deployment{&github.Deployment{ID: c.ID, Creator: c.creator}}, nil
}

func (c *GetGithubCreatorTestClient) GetUser(login string) (*github.User, error) {
	return c.creator, nil
}

func TestDeploymentGetGithubCreator(t *testing.T) {
	id := int64(1)
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	environment := "testenv"
	creator := &github.User{}
	c := &GetGithubCreatorTestClient{ID: &id, creator: creator}
	d := &Deployment{GithubID: &id, Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha, Environment: &environment}
	creator, err := d.GetGithubCreator(c)
	if err != nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %v", d, err, nil)
	}
	if creator == nil {
		t.Errorf("(%v) GetGithubCreator() = %v, _, wanted %v", d, err, creator)
	}
}

func TestDeploymentGetGithubCreatorNilOwner(t *testing.T) {
	id := int64(1)
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	environment := "testenv"
	creator := &github.User{}
	c := &GetGithubCreatorTestClient{ID: &id, creator: creator}
	d := &Deployment{GithubID: &id, Owner: nil, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha, Environment: &environment}
	_, err := d.GetGithubCreator(c)
	if err == nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetGithubCreatorNilRepo(t *testing.T) {
	id := int64(1)
	owner := "test"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	environment := "testenv"
	creator := &github.User{}
	c := &GetGithubCreatorTestClient{ID: &id, creator: creator}
	d := &Deployment{GithubID: &id, Owner: &owner, Repository: nil, SHA: &sha, PreviousSHA: &previoussha, Environment: &environment}
	_, err := d.GetGithubCreator(c)
	if err == nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetGithubCreatorNilGithubID(t *testing.T) {
	id := int64(1)
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	environment := "testenv"
	creator := &github.User{}
	c := &GetGithubCreatorTestClient{ID: &id, creator: creator}
	d := &Deployment{GithubID: nil, Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha, Environment: &environment}
	_, err := d.GetGithubCreator(c)
	if err == nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetGithubCreatorNilEnvironment(t *testing.T) {
	id := int64(1)
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	creator := &github.User{}
	c := &GetGithubCreatorTestClient{ID: &id, creator: creator}
	d := &Deployment{GithubID: &id, Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha, Environment: nil}
	_, err := d.GetGithubCreator(c)
	if err == nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetGithubCreatorDeploymentNilID(t *testing.T) {
	id := int64(1)
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	environment := "testenv"
	creator := &github.User{}
	c := &GetGithubCreatorTestClient{ID: nil, creator: creator}
	d := &Deployment{GithubID: &id, Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha, Environment: &environment}
	_, err := d.GetGithubCreator(c)
	if err == nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetGithubCreatorDeploymentNilStatus(t *testing.T) {
	id := int64(1)
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	environment := "testenv"
	c := &GetGithubCreatorTestClient{ID: &id, creator: nil}
	d := &Deployment{GithubID: &id, Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha, Environment: &environment}
	_, err := d.GetGithubCreator(c)
	if err == nil {
		t.Errorf("(%v) GetGithubCreator() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

type GetCommitsComparisonTestClient struct {
	commitsComparison *github.CommitsComparison
}

func (c *GetCommitsComparisonTestClient) CompareRepositoryCommits(owner, repo, previousSHA, SHA string) (*github.CommitsComparison, error) {
	return c.commitsComparison, nil
}

func TestDeploymentGetCommitsComparison(t *testing.T) {
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonTestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha}
	cc, err := d.GetCommitsComparison(c)
	if err != nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %v", d, err, nil)
	}
	if cc != commitsComparison {
		t.Errorf("(%v) GetCommitsComparison() = %v, _, wanted %v", d, cc, commitsComparison)
	}
}

func TestDeploymentGetCommitsComparisonNilSHA(t *testing.T) {
	owner := "test"
	repo := "foo"
	previoussha := "fedcba0987654321"
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonTestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: &owner, Repository: &repo, SHA: nil, PreviousSHA: &previoussha}
	_, err := d.GetCommitsComparison(c)
	if err == nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetCommitsComparisonNilOwner(t *testing.T) {
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonTestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: nil, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha}
	_, err := d.GetCommitsComparison(c)
	if err == nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %T", d, err, errors.New(""))
	}
}

func TestDeploymentGetCommitsComparisonNilRepo(t *testing.T) {
	owner := "test"
	sha := "1234567890abcdef"
	previoussha := "fedcba0987654321"
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonTestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: &owner, Repository: nil, SHA: &sha, PreviousSHA: &previoussha}
	_, err := d.GetCommitsComparison(c)
	if err == nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %T", d, err, errors.New(""))
	}

}

type TestDeploymentGetCommitsComparisonPreviousSHAError struct {
	previousSHA string
	SHA         string
}

func (e *TestDeploymentGetCommitsComparisonPreviousSHAError) Error() string {
	return fmt.Sprintf("SHA validation error SHA: %s Previous SHA: %s", e.SHA, e.previousSHA)
}

type GetCommitsComparisonEmptyNilPreviousSHATestClient struct {
	commitsComparison *github.CommitsComparison
}

func (c *GetCommitsComparisonEmptyNilPreviousSHATestClient) CompareRepositoryCommits(owner, repo, previousSHA, SHA string) (*github.CommitsComparison, error) {
	if previousSHA != SHA {
		return nil, &TestDeploymentGetCommitsComparisonPreviousSHAError{previousSHA: previousSHA, SHA: SHA}
	}
	return c.commitsComparison, nil
}

func TestDeploymentGetCommitsComparisonNilPreviousSHA(t *testing.T) {
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonEmptyNilPreviousSHATestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: nil}
	_, err := d.GetCommitsComparison(c)
	if err != nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %T", d, err, nil)
	}
}

func TestDeploymentGetCommitsComparisonEmptyPreviousSHA(t *testing.T) {
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := ""
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonTestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha}
	_, err := d.GetCommitsComparison(c)
	if err != nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %T", d, err, nil)
	}
}

func TestDeploymentGetCommitsComparisonBlankPreviousSHA(t *testing.T) {
	owner := "test"
	repo := "foo"
	sha := "1234567890abcdef"
	previoussha := "                "
	commitsComparison := &github.CommitsComparison{}
	c := &GetCommitsComparisonTestClient{commitsComparison: commitsComparison}
	d := &Deployment{Owner: &owner, Repository: &repo, SHA: &sha, PreviousSHA: &previoussha}
	_, err := d.GetCommitsComparison(c)
	if err != nil {
		t.Errorf("(%v) GetCommitsComparison() = _, %v, wanted %T", d, err, nil)
	}
}

func TestDeploymentToString(t *testing.T) {
	description := "test"
	d := &Deployment{Description: &description}
	if d.ToString(d.Description) != description {
		t.Errorf("ToString failed - case 1")
	}
	if d.ToString(d.JenkinsJob) != "" {
		t.Errorf("ToString failed - case 2")
	}
}

func TestDeploymentToInt(t *testing.T) {
	var id int64
	id = 1000
	d := &Deployment{ID: &id}
	if d.ToInt64(d.ID) != 1000 {
		t.Errorf("ToInt failed - case 1")
	}
	if d.ToInt64(d.Link) != 0 {
		t.Errorf("ToInt failed - case 2")
	}
}

func TestDeploymentUnixTime(t *testing.T) {
	createdat := time.Now()
	d := &Deployment{CreatedAt: &createdat}
	if d.UnixTime(d.CreatedAt) != createdat.Unix() {
		t.Errorf("UnixTime failed - case 1")
	}
	if d.UnixTime(d.UpdatedAt) != 0 {
		t.Errorf("UnixTime failed - case 2")
	}
}
