package deployment

import (
	"context"
	"encoding/json"
	"fmt"
	"path"
	"strconv"
	"strings"
	"sync"
	"time"

	"code.justin.tv/common/golibs/statsd"
	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"code.justin.tv/dta/skadi/pkg/helpers"
	jenkins "code.justin.tv/release/jenkins-api"
	twapi "code.justin.tv/release/twitch-consul-api"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
)

const (
	PendingFolder  = "pending-jobs"
	HostsEnvLength = (65536 - 1)
)

// Deployer interface allows mocking for tests
type Deployer interface {
	Deploy(*Event) error
	Resume(string) error
	Reset()
	GetStartTime() time.Time // Due to name conflict with the variable we need to set Getters()
	GetCurrentState() string
}

// JenkinsDeployer contains all external clients needed for a hubot-driven deployment.
type JenkinsDeployer struct {
	// Publicly defined variables are defined in that way in order to let json.Marshal() expose them for
	// the state-recovery mechanism. Not meant to be modified by user logic outside of JenkinsDeployer.
	GithubClient  *github.Client  `json:"-"`
	JenkinsClient *jenkins.Client `json:"-"`
	StatsdClient  statsd.Stats    `json:"-"`

	// State that is retrieved from external-systems at setup() stage.
	Config      *api.DeployConfig
	Hosts       []string          // extracted from dcsToNodes
	Link        int64             // parent deploy link, if greater than 0
	Datacenters []string          // a list of datacenters that have target hosts + user defined datacenters
	ConsulDcs   []string          // a list of datacenters to update version info
	DcsToNodes  []twapi.DCtoNodes // a list of datacenters and target hosts

	// Internal state.
	CurrentState    string
	CurrentEvent    *Event
	Job             string
	CompareLink     string
	QueueItemNumber int
	StartTime       time.Time
	URL             string
	ConsulPrefix    string `json:"-"`

	// Private variables which don't need to be serialized.
	tracking      chan *jenkins.Build
	buildError    chan error
	oneWaitForJob *sync.Once
	build         *jenkins.Build // used in WaitForJob() to work with resume and normal deployment cases
	consulTimer   time.Duration
}

func NewJenkinsDeployer(
	githubClient *github.Client,
	jenkinsClient *jenkins.Client,
	statsdClient statsd.Stats,
	consulPrefix string,
) *JenkinsDeployer {

	d := &JenkinsDeployer{
		GithubClient:  githubClient,
		JenkinsClient: jenkinsClient,
		StatsdClient:  statsdClient,
		ConsulPrefix:  consulPrefix,
	}
	// Set default values using reset
	d.Reset()
	return d
}

// Reset will reset any settings that are different per deploy.
func (d *JenkinsDeployer) Reset() {
	d.Hosts = nil
	d.Link = 0
	d.Datacenters = nil
	d.ConsulDcs = nil
	d.DcsToNodes = nil

	d.tracking = make(chan *jenkins.Build)
	d.buildError = make(chan error)
	d.oneWaitForJob = &sync.Once{}

	d.CurrentState = "setup"
	d.CurrentEvent = nil
	d.URL = ""
}

// Deploy queries consul for the appropriate hosts, and sends them to the jenkins deploy job, reporting
// progress and failures to hipchat.
func (d *JenkinsDeployer) Deploy(event *Event) error {
	var err error
	var builderr error

	if event == nil {
		return fmt.Errorf("event is nil")
	}

	d.CurrentEvent = event

	// Create a new deploy will fail if there's any ongoing deploy job on the same target.
	// when error messsage contains 'duplicate', deploy job should be retried from the caller.
	if err = d.createDbDeployStatus(); err != nil {
		return err
	}

	// Record the start of deployment. This counter will increase both in normal and resume.
	d.sendStats("start")

	for {
		switch d.CurrentState {
		// The default state is set by NewJenkinsDeployer is "setup".
		case "setup":
			err = d.setup()
			if err != nil {
				d.updateState(StateFailure)
				builderr = err
				continue
			}

			fallthrough
		case "queue-item":
			d.updateState("queue-item")

			err = d.queueItem()
			if err != nil {
				d.updateState(StateFailure)
				builderr = err
				continue
			}

			fallthrough
		case "wait-to-announce":
			d.updateState("wait-to-announce")

			// wait-to-announce and wait-to-finish are a unique set of states. Both
			// require that a background go routine waitForJob is running. We have
			// an idempotent  function that will start only one go routine per
			// event. We then wait to announce to hipchat and github that the job
			// is started before going on and waiting for the job to finish.
			d.startWaitForJob()
			err = d.waitToAnnounce()
			if err != nil {
				log.Errorf("Failed at wait-to-announce QueueID:%v - %v", d.QueueItemNumber, err)
				d.updateState(StateFailure)
				builderr = err
				d.waitToFinish() // catch any missed build errors
				continue
			}

			fallthrough
		case "wait-to-finish":
			d.updateState("wait-to-finish")

			// startWaitForJob will attempt to start waitForJob if it wasn't
			// started in the "wait-to-announce" state. (ex: a job resumed from this state)
			d.startWaitForJob()
			var build *jenkins.Build
			// build info and the error return are separate as build failure is reported as an error
			build, err = d.waitToFinish()
			if build == nil {
				if err == nil { // this case should never met
					err = fmt.Errorf("Job failed with unknown reason")
				}
				log.Errorf("Failed at wait-to-finish QueueID:%v - %v", d.QueueItemNumber, err)
				d.updateState(StatePending)
				builderr = err
				continue
			}
			if err != nil || build.Result != "SUCCESS" {
				d.updateState(StateFailure)
				builderr = fmt.Errorf("Job failed with result: %v. Link: %v", build.Result, build.URL)
				continue
			}

			// job finished with success state
			fallthrough
		case StateSuccess:
			d.updateState(StateSuccess)
			if err := d.updateDeployStatus(d.CurrentEvent, d.CurrentState, d.URL, ""); err != nil {
				log.Errorf("Got error when attempting to mark deploy a success: %v", err)
				d.updateState(StateFailure)
				continue
			}

			d.sendStats(StateSuccess)
			fallthrough
		case "end":
			d.updateState("end")
			d.deleteDbDeployStatus()

			return err
		case StateFailure:
			// state: failure
			// Mark the job as a failure in:
			// * Github
			// * Skadi logs
			d.updateState(StateFailure)
			if builderr == nil {
				builderr = fmt.Errorf("Ended up in the failure state with no error?")
			}

			if err := d.updateDeployStatus(d.CurrentEvent, d.CurrentState, d.URL, builderr.Error()); err != nil {
				log.Errorf("Got error when attempting to mark deploy a failure: %v", err)
			}

			if err := db.AppendLogGithub(fmt.Sprintf("Unignorable error occoured - %v", builderr.Error()),
				d.CurrentEvent.Deployment, d.CurrentEvent.Repository); err != nil {
				log.Errorf("Got error when attempting to append git log: %v", err)
			}

			log.Errorf("Jenkins build Error on QueueID:%v - %v", d.QueueItemNumber, builderr.Error())

			d.sendStats(StateFailure)
			d.updateState("end")
			continue
		case StatePending:
			d.updateState(StatePending)
			d.updateDeployStatus(d.CurrentEvent, d.CurrentState, d.URL, builderr.Error())
			log.Errorf("Jenkins state Unknown on QueueID:%v - %v", d.QueueItemNumber, builderr.Error())

			d.sendStats(StatePending)
			d.updateState("end")
			continue
		default:
			// This shouldn't be reachable.
			err = fmt.Errorf("Hit unknown state: %q", d.CurrentState)
			log.Println(err.Error())
			d.updateState("end")
			continue
		}
	}
}

func (d *JenkinsDeployer) Resume(deployerContext string) error {
	if err := json.Unmarshal([]byte(deployerContext), d); err != nil {
		return err
	}
	return d.Deploy(d.CurrentEvent)
}

func (d *JenkinsDeployer) GetStartTime() time.Time {
	return d.StartTime
}

func (d *JenkinsDeployer) GetCurrentState() string {
	return d.CurrentState
}
func (d *JenkinsDeployer) queueItem() error {
	var err error

	params := make(map[string]string)
	params["GIT_COMMIT"] = *d.CurrentEvent.Deployment.SHA
	params["ENVIRONMENT"] = *d.CurrentEvent.Deployment.Environment
	params["HOSTS"] = ""
	fullhost, listhost := joinAndSplitHosts(d.Hosts, HostsEnvLength)
	for i, hosts := range listhost {
		paramName := "HOSTS"
		if i > 0 {
			paramName += fmt.Sprintf("%d", i+1)
		}
		params[paramName] = hosts
	}
	// params["BRANCH"] = *d.CurrentEvent.Deployment.
	if d.Config.Smoca != nil {
		triggerSmoca := "false"
		payload, err := ParsePayload(d.CurrentEvent.Deployment.Payload)
		if err != nil {
			return fmt.Errorf("Unable to parse deployment payload: %v", err)
		}
		if payload.TriggerSmoca != "" {
			triggerSmoca = payload.TriggerSmoca
		}
		params["GITHUB_CREATOR"] = *d.CurrentEvent.Deployment.Creator.Login
		params["TRIGGER_SMOCA"] = triggerSmoca
	}
	accountID := (*d.Config.Environments)[*d.CurrentEvent.Deployment.Environment].AWSAccountID
	log.WithFields(log.Fields{"func": "JenkinsDeployer::queueItem"}).Debugf("environment: %v  accountID: %v", *d.CurrentEvent.Deployment.Environment, accountID)
	if accountID != nil {
		nodes, err := json.Marshal(d.DcsToNodes)
		if err != nil {
			return fmt.Errorf("Could not marshal dcsToNodesList: %+v to json, got error: %v", d.DcsToNodes, err)
		}
		params["ACCOUNT_ID"] = *accountID
		params["REPO"] = *d.CurrentEvent.Repository.Name
		params["OWNER"] = *d.CurrentEvent.Repository.Owner.Login
		params["ENVIRONMENT"] = *d.CurrentEvent.Deployment.Environment
		params["NODES_DC"] = string(nodes)
		params["TRIGGERED_BY"] = *d.CurrentEvent.Deployment.Creator.Login
	}

	deploy, err := FindDeployment(*d.CurrentEvent.Deployment.ID)
	if err == nil {
		params["BRANCH"] = *deploy.Branch
		params["SKADI_ID"] = strconv.FormatInt(int64(*deploy.ID), 10)
	} else {
		log.Warnf("skadi deployment for github deployment %d could not be found: %v", *d.CurrentEvent.Deployment.ID, err)
	}
	log.Printf("Triggering Jenkins job: %v, with params: %+v", d.Job, params)

	err = helpers.Retry(func() error {
		d.QueueItemNumber, err = d.JenkinsClient.BuildWithParameters(d.Job, params)
		if err != nil {
			return fmt.Errorf("Unable to start job %q: %v", d.Job, err)
		}
		return nil
	})
	if err != nil {
		return err
	}

	log.Printf("Jenkins job:%v with params: %+v was triggered successfully. QueueID:%v", d.Job, params, d.QueueItemNumber)
	db.UpdateDeploymentWithJenkinsParams(*d.CurrentEvent.Deployment.ID, d.Job, fullhost)

	return nil
}

func (d *JenkinsDeployer) setup() error {
	d.StartTime = time.Now()

	// We should always fetch settings from the head the default branch. That
	// way old versions of code can be deployed even if the settings have had
	// to change.
	deployConfig, err := config.LoadDeployConfig(
		d.GithubClient,
		consulClient,
		*d.CurrentEvent.Repository.Owner.Login,
		*d.CurrentEvent.Repository.Name,
		githubcache.GetDefaultBranch(*d.CurrentEvent.Repository.Owner.Login, *d.CurrentEvent.Repository.Name),
	)
	if err != nil {
		return err
	}
	d.Config = deployConfig

	// Get Jenkins deploy job name
	d.Job, err = d.Config.GetDeployJob(*d.CurrentEvent.Deployment.Environment)
	if err != nil {
		return err
	}

	enableVersionUpdate := true
	enableConsulSearch := false
	if d.Config.ConsulServices != nil && len(*d.Config.ConsulServices) > 0 {
		enableConsulSearch = true
	}

	// Check hosts fields to see if this is targeted deploy.
	deployment, err := db.FindDeploymentByGithubID(*d.CurrentEvent.Deployment.ID)
	if err == nil && deployment != nil {
		if deployment.Link != nil && *deployment.Link > 0 {
			d.Link = *deployment.Link
			db.AppendLog(fmt.Sprintf("Targeted redeploy created from deploy %d", d.Link), *deployment.ID)
		}
		if deployment.Hosts != nil && *deployment.Hosts != "" {
			if d.Hosts = helpers.UniqueStringsSorted(helpers.SplitStringNoEmpty(*deployment.Hosts, ",")); len(d.Hosts) == 0 {
				return fmt.Errorf("Invalid format found with hosts field - %v", *deployment.Hosts)
			}
			enableConsulSearch = false
			enableVersionUpdate = false
			db.AppendLog(fmt.Sprintf("Target hosts: %v (given - skip consul search)", d.Hosts), *deployment.ID)
			// log to the parent deploy
			if d.Link > 0 {
				db.AppendLog(fmt.Sprintf("Targeted redeploy %d created for hosts: %v", *deployment.ID, d.Hosts), d.Link)
			}
		}
	}

	// Lookup the target hosts if this is courier-based deployment
	if enableConsulSearch {
		d.Datacenters = consulDatacenters
		if d.Config.Datacenters != nil {
			// Use user-defined datacenters
			if len(*d.Config.Datacenters) > 0 {
				d.Datacenters = helpers.UniqueStringsSorted(*d.Config.Datacenters)
			}
		}
		db.AppendLogGithub(fmt.Sprintf("Target datacenters: %v", d.Datacenters),
			d.CurrentEvent.Deployment, d.CurrentEvent.Repository)

		// Start timing the total time it takes for consul to lookup hosts
		start := time.Now()

		db.AppendLogGithub(fmt.Sprintf("Searching hosts in consul for services %v with tag '%v'", *d.Config.ConsulServices, *d.CurrentEvent.Deployment.Environment),
			d.CurrentEvent.Deployment, d.CurrentEvent.Repository)
		dcsToNodes, err := fsConsulClient.GetAliveHostnamesWithDCs(*d.Config.ConsulServices, *d.CurrentEvent.Deployment.Environment, d.Datacenters)
		if err != nil {
			return err
		}

		// Stop timing the time it took for consul to lookup hosts
		d.consulTimer = time.Since(start)

		// Store results in member variables
		d.Hosts = helpers.GetHostnamesFromDCtoNodes(dcsToNodes)
		d.DcsToNodes = dcsToNodes

		// Make an informative string with number of target hosts per dc
		hostCntPerDcs := []string{}
		for _, dcToNode := range dcsToNodes {
			if len(dcToNode.Nodes) == 0 {
				continue
			}
			hostCntPerDcs = append(hostCntPerDcs, []string{fmt.Sprintf("%v(%v)", dcToNode.DC, len(dcToNode.Nodes))}...)
		}
		db.AppendLogGithub(fmt.Sprintf("Total %v hosts found in datacenters: %v", len(d.Hosts), hostCntPerDcs),
			d.CurrentEvent.Deployment, d.CurrentEvent.Repository)
		db.AppendLogGithub(fmt.Sprintf("Target hosts: %v", d.Hosts),
			d.CurrentEvent.Deployment, d.CurrentEvent.Repository)
	}

	// Skip consul update if this is targeted deploy
	if enableVersionUpdate == false {
		return nil
	}

	// a list of datacenters that must have the consul version updated successfully
	d.ConsulDcs = helpers.UniqueStringsSorted(append(d.Datacenters, consulMasterDatacenters...))

	// Update deployed version, so that actual deploy tools such as courier know what version to deploy
	err = d.updateConsulVersion(ConsulPrefixDeployedVersion, d.CurrentEvent, d.ConsulDcs)
	if err != nil {
		log.Printf("Error updating deployed version to consul: %v", err)
		return err
	}

	return nil
}

// startWaitForJob is a function meant to be run in a separate go routine. It
// will check if it has been started already by checking if the builError
// channel exists. If it doesn't already exist it will build the channel and
// then wait for waitForJob. There is a sister function called waitForFinish
// which is blocking on this to finish.
func (d *JenkinsDeployer) startWaitForJob() {
	// Wrap our waitForJob in a sync.Once
	go d.oneWaitForJob.Do(func() {
		defer close(d.buildError)

		// WaitForJob() returns FAILURE state as error, we ignore this error when it returns build info
		// to distinguish between actual failed build and other real errors for better handling.
		build, err := d.JenkinsClient.WaitForJob(d.Job, d.QueueItemNumber, d.tracking)
		if build != nil {
			d.build = build
			d.URL = build.URL
			log.Printf("build info QueueID:%v - %v", d.QueueItemNumber, build)
		}

		if err != nil {
			d.buildError <- err
			log.Errorf("build error QueueID:%v - %v", d.QueueItemNumber, err)
		}
	})
}

// waitToAnnounce will wait for a message on the tracking channel and then announce the start.
func (d *JenkinsDeployer) waitToAnnounce() error {
	build, ok := <-d.tracking
	if !ok {
		return fmt.Errorf("Did not get build from jenkins. QueueID:%v", d.QueueItemNumber)
	}

	log.Printf("Jenkins job %v started. QueueID:%v", build.Number, d.QueueItemNumber)

	duration, err := time.ParseDuration(fmt.Sprintf("%vs", build.EstimatedDuration/1000))
	if err != nil {
		log.Warnf("failed to parse deploy duration, but continue - %v", err)
	}

	err = d.updateDeployStatus(d.CurrentEvent, StatePending, build.URL, fmt.Sprintf("Expected duration: %v", duration.String()))
	if err != nil {
		log.Warnf("failed to update deploy status, but continue - %v", err)
	}

	err = db.AppendLogGithub(fmt.Sprintf("Jenkins job %v started", build.Number), d.CurrentEvent.Deployment, d.CurrentEvent.Repository)
	if err != nil {
		log.Warnf("failed to update github log, but continue - %v", err)
	}
	return nil
}

// Wait for waitForJob to finish.
func (d *JenkinsDeployer) waitToFinish() (*jenkins.Build, error) {
	// For normal deployments, previous build result will be set here and rest of range loops will just bypass.
	build := d.build
	for b := range d.tracking {
		// this is a resume deployment case which started from wait-to-finish stage
		build = b
	}
	var err error
	for e := range d.buildError {
		// this is a resume deployment case which started from wait-to-finish stage
		err = e
	}

	return build, err
}

func (d *JenkinsDeployer) updateDeployStatus(event *Event, state string, deployLink string, desc string) error {
	req := &github.DeploymentStatusRequest{
		State:  &state,
		LogURL: &deployLink,
	}
	if desc != "" {
		req.Description = &desc
	}

	// In order to prevent a race condition with `state mismatch detector` in status.go,
	// database must be updated before git.
	err := db.UpdateDeploymentState(*event.Deployment.ID, state)
	if err != nil {
		e := fmt.Errorf("deploy %v succeeded, but failed to update state in postgres: %v", *event.Deployment.ID, err)
		log.Errorln(e.Error())
		return e
	}

	_, _, err = d.GithubClient.Repositories.CreateDeploymentStatus(context.TODO(), *event.Repository.Owner.Login, *event.Repository.Name, *event.Deployment.ID, req)
	if err != nil {
		log.Errorf("deploy %v succeeded, but failed to update status : %v", *event.Deployment.ID, err)
	}

	if state == StateSuccess {
		err = d.updateConsulVersion(ConsulPrefixKnownGoodVersion, event, d.ConsulDcs)
		if err != nil {
			e := fmt.Errorf("deploy %v succeeded, but failed to update known good version - %v", *event.Deployment.ID, err)
			log.Errorln(e.Error())
			return e
		}
	}
	if desc != "" {
		err = db.AppendLogGithub(desc, d.CurrentEvent.Deployment, d.CurrentEvent.Repository)
		if err != nil {
			return err
		}
	}

	return nil
}

func (d *JenkinsDeployer) updateState(state string) error {
	if d.CurrentState == state {
		// duplicated calls are intended by the logic
		return nil
	}
	d.CurrentState = state
	if err := d.updateDbDeployStatus(); err != nil {
		log.Errorf("Error updating deploy status. GitID:%v - %v", *d.CurrentEvent.Deployment.ID, err)
		return err
	}
	return nil
}

func (d *JenkinsDeployer) deployTargetID() string {
	return path.Join(*d.CurrentEvent.Repository.FullName, *d.CurrentEvent.Deployment.Environment)
}

func (d *JenkinsDeployer) createDbDeployStatus() error {
	target := d.deployTargetID()
	log.Printf("Creating deploy status. GitID:%v, TargetID:%v, State:%v", *d.CurrentEvent.Deployment.ID, target, d.CurrentState)
	deployerJson, err := json.Marshal(d)
	if err != nil {
		// This error can't happen.
		log.Errorf("Failed to marshall deploy job, GitID:%v, TargetID:%v - %v", *d.CurrentEvent.Deployment.ID, target, err)
		return err
	}
	if err = db.CreateDeploymentStatus(target, d.CurrentState, string(deployerJson)); err != nil {
		if strings.Contains(err.Error(), "duplicate") {
			// When deploy job is started, it's clear that there's no deploy job running with this target
			// on local machine, so try to remove mine if there's any residue left over for some reason
			// and try again. This is a safety logic to filter out any possible stalled job.
			log.Warnf("Possible BUG, CreateDeploymentStatus returned duplicate error. GitID:%v, TargetID:%v - %v", *d.CurrentEvent.Deployment.ID, target, err)
			db.DeleteMyDeploymentStatus(target)
		}
		if err = db.CreateDeploymentStatus(target, d.CurrentState, string(deployerJson)); err != nil {
			log.Errorf("Failed to create deploy status. GitID:%v, TargetID:%v, State:%v - %v", *d.CurrentEvent.Deployment.ID, target, d.CurrentState, err)
			return err
		}
	}

	return nil
}

func (d *JenkinsDeployer) updateDbDeployStatus() error {
	target := d.deployTargetID()
	log.Printf("Updating deploy status. GitID:%v, TargetID:%v, State:%v", *d.CurrentEvent.Deployment.ID, target, d.CurrentState)
	deployerJson, err := json.Marshal(d)
	if err != nil {
		// This error can't happen.
		log.Errorf("Failed to marshall deploy job, GitID:%v, TargetID:%v - %v", *d.CurrentEvent.Deployment.ID, target, err)
		return err
	}
	if err = db.UpdateMyDeploymentStatus(target, d.CurrentState, string(deployerJson)); err != nil {
		log.Errorf("Failed to update deploy status. GitID:%v, TargetID:%v, State:%v - %v", *d.CurrentEvent.Deployment.ID, target, d.CurrentState, err)
		return err
	}
	return nil
}

func (d *JenkinsDeployer) deleteDbDeployStatus() error {
	target := d.deployTargetID()
	log.Printf("Deleting deploy status. GitID:%v, TargetID:%v", *d.CurrentEvent.Deployment.ID, target)
	if err := db.DeleteMyDeploymentStatus(target); err != nil {
		log.Errorf("Failed to delete deploy status. GitID:%v, TargetID:%v - %v", *d.CurrentEvent.Deployment.ID, target, d.CurrentState)
		return err
	}
	return nil
}

func (d *JenkinsDeployer) updateConsulVersion(prefix string, event *Event, datacenters []string) error {
	if prefix == "" || event == nil {
		return fmt.Errorf("Invalid arguments.")
	}

	if datacenters == nil {
		return nil
	}

	db.AppendLogGithub(fmt.Sprintf("Updating %v to %v in datacenters: %v", prefix, *event.Deployment.SHA, datacenters),
		d.CurrentEvent.Deployment, d.CurrentEvent.Repository)

	var wait sync.WaitGroup
	failedDcsChan := make(chan string, len(datacenters))
	key := fmt.Sprintf("%v/%v/%v", prefix, *event.Repository.FullName, *event.Deployment.Environment)
	for _, dc := range datacenters {
		wait.Add(1)
		go func(dc string) {
			defer wait.Done()
			cinfo.IncreaseCounter("KeyPut")
			err := helpers.Retry(func() error {
				return fsConsulClient.UpdateKV(key, *event.Deployment.SHA, dc)
			})
			if err != nil {
				log.Errorf("Failed to update %v for datacenter %v - %v", key, dc, err)
				failedDcsChan <- dc
			}
		}(dc)
	}

	wait.Wait()
	close(failedDcsChan)
	failedDcs := helpers.SerializeStringChan(&failedDcsChan)

	// Any failure in given datacenters here is critical for the sanity of the deployment.
	if len(failedDcs) > 0 {
		err := fmt.Errorf("Failed to update %v in datacenters: %v", prefix, failedDcs)
		db.AppendLogGithub(fmt.Sprintf("CAUTION: %v - critical", err),
			d.CurrentEvent.Deployment, d.CurrentEvent.Repository)
		return err
	}

	return nil
}

func (d *JenkinsDeployer) sendStats(status string) {
	err := d.sendToStatsd(status)
	if err != nil {
		log.Warnf("failed to report to statsd - %v", err)
	}
}

func (d *JenkinsDeployer) sendToStatsd(status string) error {
	if d.CurrentEvent.Repository == nil ||
		d.CurrentEvent.Repository.Owner == nil {
		return fmt.Errorf("Missing data needed to log deploy in statsd")
	}

	// Bucket looks like 'deployment.dta.skadi.production.success'
	bucket := fmt.Sprintf("%s.%s.%s.%s.%s",
		statsNamePrefix,
		*d.CurrentEvent.Repository.Owner.Login,
		*d.CurrentEvent.Repository.Name,
		*d.CurrentEvent.Deployment.Environment,
		status,
	)
	// This is a seprate bucket because it's slightly different for the consul_lookup
	// Bucket looks like 'deployment.dta.skadi.production.consul_lookup'
	consul_bucket := fmt.Sprintf("%s.%s.%s.%s.consul_lookup",
		statsNamePrefix,
		*d.CurrentEvent.Repository.Owner.Login,
		*d.CurrentEvent.Repository.Name,
		*d.CurrentEvent.Deployment.Environment,
	)

	// Increment a counter for the build
	d.StatsdClient.Incr(bucket)

	// Record the time it took to build
	d.StatsdClient.Timing(bucket, time.Since(d.StartTime))

	// Record the time it took to do a consul lookup for a specific build
	d.StatsdClient.Timing(consul_bucket, d.consulTimer)

	// Record the time it took to do a consul lookup for all builds
	d.StatsdClient.Timing(statsNamePrefix+".total.consul_lookup", d.consulTimer)

	if productionFlag && status == StateSuccess {
		// Backwards compatibility with old naming scheme.
		// Bucket looks like 'release.skadi.production'
		bucket := fmt.Sprintf("%s.%s.%s",
			*d.CurrentEvent.Repository.Owner.Login,
			*d.CurrentEvent.Repository.Name,
			*d.CurrentEvent.Deployment.Environment,
		)
		d.StatsdClient.Incr(bucket)
		d.StatsdClient.Timing(bucket, time.Since(d.StartTime))
	}

	return nil
}

// joinAndSplitHosts returns comma-separated hosts in single string and string array bounded by maxlen.
// When 0 length `hosts` is given, it will return []string{""}
//   [Example]
//     fmt.Println(joinAndSplitHosts([]string{"Hello","DTA","How","are","u","doing"}, 10))
//   [Output]
//     Hello,DTA,How,are,u,doing [Hello,DTA How,are,u doing]
func joinAndSplitHosts(hosts []string, maxlen int) (string, []string) {
	list := []string{}
	join := strings.Join(hosts, ",")
	head, tail, over := 0, 0, maxlen
	for i, c := range join {
		if c == ',' {
			tail = i
		}
		if i >= over {
			if head == tail {
				// this won't happen unless the length of single host exceeds maxlen
				// anyway in this case, we allow overflow cause we can't cut hostname in the middle.
				continue
			}
			list = append(list, join[head:tail])
			tail++
			head = tail
			over = head + maxlen
		}
	}
	list = append(list, join[head:])
	return join, list
}
