package deployment

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/dta/skadi/pkg/build"
	"code.justin.tv/dta/skadi/pkg/helpers"
	"code.justin.tv/release/jenkins-api"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
)

const (
	// Following are state info we record to Git and DB.
	// The state that are not defined here such as 'wait-to-announce' are internal state.
	StateSuccess = "success"
	StatePending = "pending"
	StateError   = "error"
	StateFailure = "failure"
	StateUnknown = "unknown" // This is one of final state meaning we'll no more try to check the state.

	ElapsedTimeInPendingForStallDetection = time.Minute * 30

	JenkinsResultSuccess = "SUCCESS"
	JenkinsResultFailure = "FAILURE"
	JenkinsResultAborted = "ABORTED"
)

// Status is the current state of a particular deployment
type Status struct {
	ID           int          `json:"id"`
	State        *string      `json:"state,omitempty"`
	Description  *string      `json:"description"`
	TargetURL    *string      `json:"target_url"`
	UpdatedAt    *time.Time   `json:"updated_at,omitempty"`
	Percent      int          `json:"percent"`
	NodeTally    string       `json:"nodeTally"`
}

func isFinalState(state string) bool {
	return (state == StateSuccess ||
		state == StateFailure ||
		state == StateError ||
		state == StateUnknown)
}

func isFinalGitState(state string) bool {
	return (state == StateSuccess ||
		state == StateFailure ||
		state == StateError)
}

// GetStatus returns deploy status.
// [DTA-1536]
//   Deploy state info in database is considered as origin.  If mismatch state info found between database and git,
//   Git status will be updated with db's info when it's unclear who's right and by the logic chage of status update,
//   Git's status info can't be updated without prior successful db info update.
func GetStatus(client *github.Client, jenkinsClient *jenkins.Client, deployID int) (*Status, error) {
	var owner string
	var name string
	var githubID int64
	var dbstate string
	var updatedat time.Time

	err := db.QueryRow("SELECT githubid, owner, repository, state, updatedat FROM deployments WHERE id=$1", deployID).Scan(&githubID, &owner, &name, &dbstate, &updatedat)
	if err != nil {
		log.Warnf("Failed to find deploy. DeployID=%v - %v", deployID, err)
		return nil, err
	}

	if dbstate == "" {
		dbstate = StateUnknown
	}

	// Check if it's about time to give up checking the state.
	doStateMismatchCorrection := false
	if !isFinalState(dbstate) && time.Since(updatedat) > ElapsedTimeInPendingForStallDetection {
		// We've been rechecking the status until at this point.
		log.Warnf("Possible stall state detected(no update since %v, state=%v). DeployID=%v", updatedat.String(), dbstate, deployID)
		defer func() {
			// This gets triggered at the end after giving a chance to check one last time.
			// If the state is not in final still, we'll assume there's permanent error and mark it as unknown.
			// Unknown is a final state so we won't check it back again.
			if !isFinalState(dbstate) {
				newstate := StateUnknown
				log.Warnf("Updating stalled job to %v from %v. DeployID=%v", newstate, dbstate, deployID)
				if err := db.UpdateDeploymentState(githubID, newstate); err != nil {
					log.Errorf("Failed to update deploy state. DeployID=%v - %v", deployID, err)
				}
			}
		}()
		doStateMismatchCorrection = true
	}

	// Build default return structure
	status := &Status{
		ID:    deployID,
		State: &dbstate,
	}

	// Fetch latest Git state
	statuses, gheResponse, err := client.Repositories.ListDeploymentStatuses(context.TODO(), owner, name, githubID, nil)
	if err != nil {
		log.Errorf("Failed to find a corresponding deployment info from Git. DeployID=%v, GitID:%v - %v", deployID, githubID, err)
		if gheResponse != nil && gheResponse.StatusCode == http.StatusNotFound {
			// This could happen when we move a repo to new place along with its deploy logs,
			// in any case we don't find the status from git and if dbstate is marked as one
			// of final states, we just honor the state info rather than returning nothing.
			if isFinalState(dbstate) {
				return status, nil
			}
			// We couldn't find the deployment, but we didn't actually error.
			return nil, nil
		}
		return nil, err
	}

	var gitstatus *github.DeploymentStatus
	for _, gitstatus = range statuses {
		// Skip inactive event.
		// See: https://developer.github.com/changes/2016-04-06-deployment-and-deployment-status-enhancements
		if gitstatus.State != nil && *gitstatus.State != "inactive" {
			// Set more data from Git deployment info
			status.Description = gitstatus.Description
			status.TargetURL = gitstatus.TargetURL
			break
		}
	}
	if helpers.IsEmptyStringp(status.TargetURL) {
		return status, nil
	}

	// Check Jenkins state
	if !isFinalState(*status.State) {
		job, buildId, err := build.ParseTargetUrl(*status.TargetURL)
		if err == nil {
			b, err := jenkinsClient.GetBuild(job, buildId)
			if err == nil {
				if doStateMismatchCorrection {
					newstate := ""
					if b.Result == JenkinsResultSuccess {
						newstate = StateSuccess
					} else if b.Result == JenkinsResultFailure || b.Result == JenkinsResultAborted {
						newstate = StateFailure
					}
					if newstate != "" && newstate != dbstate {
						log.Warnf("Updating state to %v from %v. DeployID=%v", newstate, dbstate, deployID)
						if err := db.UpdateDeploymentState(githubID, newstate); err != nil {
							log.Errorf("Failed to update deploy state. DeployID=%v - %v", deployID, err)
						}
						dbstate = newstate
					}
				}
				_, percent := build.CalculateJenkinsPercent(b)
				status.Percent = int(percent)
			} else {
				log.Warnf("Failed to get build from jenkins. DeployID=%v, Job:%v - %v", deployID, job, err)
			}
		} else {
			log.Warnf("Failed to parse target URL. DeployID=%v, TargetURL=%v - %v", deployID, *status.TargetURL, err)
		}
	} else {
		status.Percent = 100
	}

	// Check if status info in Git is behind or do corrective action if dbstate has updated above.
	if doStateMismatchCorrection && isFinalGitState(dbstate) &&
		gitstatus != nil && gitstatus.State != nil && dbstate != *gitstatus.State {
		log.Warnf("Found state mismatch, Syncing with Git. DeployID=%v, DB State=%v, Git State=%v", deployID, dbstate, *gitstatus.State)
		if _, _, err := client.Repositories.CreateDeploymentStatus(context.TODO(), owner, name, githubID, &github.DeploymentStatusRequest{
			State:  &dbstate,
			LogURL: status.TargetURL,
		}); err != nil {
			// Just log and move on, it will try again on the next call.
			log.Errorf("Failed to sync Status to Git. DeployID=%v, DB State=%v, Git State=%v - %v", deployID, dbstate, *gitstatus.State, err)
		}
	}

	return status, nil
}
