package deploystatus

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/rtevent_receiver"
	"github.com/gorilla/mux"
)

func DeployStatusPerDeployHandler(w http.ResponseWriter, r *http.Request) {
	deployid, _ := strconv.ParseUint(mux.Vars(r)["deployid"], 10, 64)

	// by the router config, it's guranteed that they have values but let just double check
	if deployid == 0 {
		config.JSONError(w, http.StatusBadRequest, "", fmt.Errorf("Invalid arguments"))
		return
	}

	deployment, err := db.GetDeployment(deployid)
	if err != nil {
		config.JSONError(w, http.StatusNotFound, "", fmt.Errorf("No deploy information found for %v", deployid))
		return
	}

	targethosts := ""
	if deployment.Hosts != nil {
		targethosts = *deployment.Hosts
	}
	hosts, err := db.GetHostsByDeployID(deployid, targethosts)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "", fmt.Errorf("%v", err))
		return
	}

	s := &DeployStatus{
		Service:     fmt.Sprintf("%s/%s", *deployment.Owner, *deployment.Repository),
		Environment: *deployment.Environment,
		Deployed: &Artifact{
			SHA:    *deployment.SHA,
			Branch: *deployment.Branch,
		},
		HostStat: calculateDeployStat(hosts, *deployment.SHA),
		Hosts:    hosts,
	}

	json.NewEncoder(w).Encode(s)
}

func calculateDeployStat(hosts []*HostVersionStatus, sha string) *HostStat {
	s := &HostStat{
		TotalHosts: len(hosts),
	}

	for _, h := range hosts {
		if h.Status == rtevent_receiver.STATUS_FAILURE {
			s.NumFailure++
		} else if h.Status == rtevent_receiver.STATUS_SUCCESS {
			if h.Sha == sha && h.Phase == rtevent_receiver.COURIER_PHASE_RESTART_REMOTE {
				s.NumSuccess++
			} else {
				s.NumProgress++
			}
		} else {
			s.NumUnknown++
		}
	}
	s.HostsMatch = s.NumSuccess
	s.HostsMismatch = s.TotalHosts - s.HostsMatch
	return s
}
