package deploystatus

import (
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/dta/skadi/pkg/config"
	"github.com/gorilla/mux"
	"net"
)

func DeployStatusPerServiceHandler(w http.ResponseWriter, r *http.Request) {
	owner := mux.Vars(r)["owner"]
	repo := mux.Vars(r)["repo"]
	env := mux.Vars(r)["env"]

	artifact, err := db.GetDeployedVersion(owner, repo, env)
	if err != nil {
		config.JSONError(w, http.StatusNotFound, "", fmt.Errorf("No deploy information found for %s/%s/%s", owner, repo, env))
		return
	}

	hosts, err := db.GetHostsByEnv(owner, repo, env)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "", fmt.Errorf("%v", err))
		return
	}

	livehosts := []*HostVersionStatus{}
	for _, h := range hosts {
		if h.Health == HostStatusAlive {
			livehosts = append(livehosts, h)
		}
	}

	s := &DeployStatus{
		Service:     fmt.Sprintf("%s/%s", owner, repo),
		Environment: env,
		Deployed:    artifact,
		HostStat:    calculateHostStat(livehosts, artifact.SHA),
		Hosts:       livehosts,
	}

	json.NewEncoder(w).Encode(s)
}

func UpdateHostVersionInfoHandler(w http.ResponseWriter, r *http.Request) {
	status := mux.Vars(r)["status"]
	owner := mux.Vars(r)["owner"]
	repo := mux.Vars(r)["repo"]
	env := mux.Vars(r)["env"]
	ip := mux.Vars(r)["ip"]

	if net.ParseIP(ip) == nil {
		config.JSONError(w, http.StatusBadRequest, "", fmt.Errorf("Invalid arguments"))
		return
	}

	if status == "moved" {
		if err := db.UpdateHostVersionInfo(owner, repo, env, ip, HOST_VERSION_STATUS_MOVED); err != nil {
			config.JSONError(w, http.StatusNotFound, "", fmt.Errorf("%v", err))
			return
		}
	} else {
		config.JSONError(w, http.StatusBadRequest, "", fmt.Errorf("Invalid arguments"))
	}
}

func calculateHostStat(hosts []*HostVersionStatus, sha string) *HostStat {
	s := &HostStat{
		TotalHosts: len(hosts),
	}

	for _, h := range hosts {
		if h.Sha == sha {
			s.HostsMatch++
		}
	}
	s.HostsMismatch = s.TotalHosts - s.HostsMatch

	s.NumSuccess = s.HostsMatch
	s.NumFailure = s.HostsMismatch
	return s
}
