package deploystatus

import (
	"fmt"
	"sync"
	"time"

	"code.justin.tv/dta/skadi/pkg/helpers"
	twapi "code.justin.tv/release/twitch-consul-api"
	log "github.com/Sirupsen/logrus"
	consulapi "github.com/hashicorp/consul/api"
)

const (
	CacheUpdateIntervalSec = 60
)

type ConsulCacheGlobal struct {
	once        sync.Once
	ipHostMap   map[string]*HostInfo // key is IP address
	nameHostMap map[string]*HostInfo // key is FQDN
}

type HostInfo struct {
	Name       string
	IPAddr     string
	FQDN       string
	Alive      bool
	Datacenter string
}

var (
	cc ConsulCacheGlobal
)

func launchConsulCacherThread(client *consulapi.Client, datacenters *string) error {
	reterr := fmt.Errorf("Already running")
	cc.once.Do(func() {
		fsConsul := helpers.NewFailSafeConsulClient(client)
		go func() {
			for {
				dcs, err := fsConsul.ListDatacenters(helpers.UniqueStringsSorted(helpers.SplitStringNoEmpty(*datacenters, ",")))
				if err == nil {
					cc.ipHostMap, cc.nameHostMap = generateIpHostMap(dcs)
				} else {
					log.Errorf("Failed to update consul cache - %v", err)
				}
				time.Sleep(CacheUpdateIntervalSec * time.Second)
			}
		}()
		reterr = nil
	})

	return reterr
}

func GetIpHostMap() map[string]*HostInfo {
	for {
		if cc.ipHostMap != nil {
			break
		}
		time.Sleep(100 * time.Millisecond)
	}
	return cc.ipHostMap
}

func GetNameHostMap() map[string]*HostInfo {
	for {
		if cc.nameHostMap != nil {
			break
		}
		time.Sleep(100 * time.Millisecond)
	}
	return cc.nameHostMap
}

func GetHostInfoByIP(ipaddr string) *HostInfo {
	return GetIpHostMap()[ipaddr]
}

func GetHostInfoByHostname(hostname string) *HostInfo {
	return GetNameHostMap()[hostname]
}

func generateIpHostMap(dcs map[string]*twapi.Datacenter) (map[string]*HostInfo, map[string]*HostInfo) {
	ipMap := make(map[string]*HostInfo)
	nameMap := make(map[string]*HostInfo)
	for dcName, dcInfo := range dcs {
		for name, IP := range dcInfo.IPs {
			hostinfo := &HostInfo{
				Name:       name,
				IPAddr:     IP,
				FQDN:       dcInfo.FQDNs[name],
				Alive:      dcInfo.Alive[name],
				Datacenter: dcName,
			}
			ipMap[IP] = hostinfo
			nameMap[hostinfo.FQDN] = hostinfo
		}
	}
	return ipMap, nameMap
}
