package deploystatus

import (
	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/rtevent_receiver"
	"fmt"
	"github.com/jmoiron/sqlx"
	"net"
	"strings"
	"time"
)

type DB struct {
	*sqlx.DB
}

const (
	HOST_VERSION_STATUS_OK    = 0
	HOST_VERSION_STATUS_MOVED = 1
)

func NewDB(dbInstance *sqlx.DB) *DB {
	return &DB{
		dbInstance,
	}
}

// Get a deployment information by deploy id
func (db *DB) GetDeployment(id uint64) (*deployment.Deployment, error) {
	var d deployment.Deployment
	err := db.Get(&d, "SELECT * FROM deployments WHERE id=$1", id)
	if err != nil {
		return nil, err
	}
	return &d, nil
}

// Get current/latest deployed-version from environment table
func (db *DB) GetDeployedVersion(owner string, repo string, env string) (*Artifact, error) {
	artifact := Artifact{}
	err := db.Get(&artifact, "SELECT deployedsha as sha, deployedbranch AS branch FROM environments WHERE owner=$1 AND repository=$2 AND environment=$3", owner, repo, env)
	if err != nil {
		return nil, err
	}
	return &artifact, nil
}

// Get a list of hosts with version status per deploy id
func (db *DB) GetHostsByDeployID(id uint64, targetHosts string) ([]*HostVersionStatus, error) {
	s := []*HostVersionStatus{}
	err := db.Select(&s, "SELECT ipaddr, hostname, sha, phase, status, detail AS desc, changedat AS updated FROM deployment_deploystatus WHERE deployment_id=$1 ORDER BY changedat", id)
	if err != nil {
		return nil, err
	}

	// Unify the events into the final status per host
	m := make(map[string]*HostVersionStatus)
	for _, h := range s {
		// Check if hostname is IP address. This happens if users converted FQDN to IP address.
		// And we keep the event as is to distinguish whether FQDN or IP address is used at the deploy time,
		// But it's desired to get the hostname on the report if possible
		if net.ParseIP(h.Hostname) != nil {
			c := GetHostInfoByIP(h.IpAddr)
			if c != nil {
				h.Hostname = c.FQDN
			}
		}

		// Since this deploy events are sent by one master-courier, the evemt order by changedat time can be
		// safely assumed as accurate, so we don't really need to check the phase but just override.
		m[h.Hostname] = h
	}

	// Combine with consul information
	for _, h := range m {
		c := GetHostInfoByHostname(h.Hostname)
		if c == nil {
			h.Health = HostStatusGone
		} else {
			h.IpAddr = c.IPAddr
			h.Datacenter = c.Datacenter
			h.Health = statusStr(c.Alive)
		}
		m[h.Hostname] = h

	}

	// Count number of target hosts
	numtargets := 0
	for _, hostname := range strings.Split(targetHosts, ",") {
		hostname = strings.TrimSpace(hostname)
		if hostname == "" {
			continue
		}
		numtargets++
	}

	// If not everybody reported, add target hosts not found from the report
	if numtargets != len(m) {
		timenow := time.Now()
		for _, hostname := range strings.Split(targetHosts, ",") {
			hostname = strings.TrimSpace(hostname)
			if hostname == "" {
				continue
			}
			if _, ok := m[hostname]; ok {
				continue
			}

			h := &HostVersionStatus{
				Hostname: hostname,
				Status:   rtevent_receiver.STATUS_UNKNOWN,
				Health:   HostStatusGone,
				Updated:  timenow,
			}
			c := GetHostInfoByHostname(hostname)
			if c != nil {
				h.IpAddr = c.IPAddr
				h.Datacenter = c.Datacenter
				h.Health = statusStr(c.Alive)
			} else {
				addrs, err := net.LookupHost(hostname)
				if err != nil && len(addrs) > 0 {
					h.IpAddr = addrs[0]
				}
			}
			m[hostname] = h

		}
	}

	// Convert to array
	uni := []*HostVersionStatus{}
	for _, v := range m {
		uni = append(uni, v)
	}

	return uni, nil
}

// Get a list of hosts with version status per environment
func (db *DB) GetHostsByEnv(owner string, repo string, env string) ([]*HostVersionStatus, error) {
	s := []*HostVersionStatus{}
	err := db.Select(&s, "SELECT ipaddr, sha, deployedat AS updated FROM host_version WHERE owner=$1 AND repository=$2 AND environment=$3 AND status=$4", owner, repo, env, HOST_VERSION_STATUS_OK)
	if err != nil {
		return nil, err
	}

	for _, h := range s {
		c := GetHostInfoByIP(h.IpAddr)
		if c == nil {
			h.Health = HostStatusGone
			continue
		}
		h.Hostname = c.FQDN
		h.Datacenter = c.Datacenter
		h.Health = statusStr(c.Alive)
	}

	return s, nil
}

// Get a matching host info from host_version table
func (db *DB) GetHostServices(ipaddr string) ([]*HostServices, error) {
	s := []*HostServices{}
	err := db.Select(&s, "SELECT concat(owner, '/', repository) AS service, environment, sha, deployedat AS updated FROM host_version WHERE ipaddr=$1 AND status=$2", ipaddr, HOST_VERSION_STATUS_OK)
	if err != nil {
		return nil, err
	}
	return s, nil
}

func (db *DB) UpdateHostVersionInfo(owner string, repo string, env string, ipaddr string, status int) error {
	result, err := db.Exec("UPDATE host_version SET status=$1, updatedat = now() WHERE owner=$2 AND repository=$3 AND environment=$4 AND ipaddr=$5", status, owner, repo, env, ipaddr)
	if err != nil {
		return err
	}

	num, err := result.RowsAffected()
	if num != 1 {
		return fmt.Errorf("Not found")
	}

	return err
}

func (db *DB) DeleteHostVersionInfo(owner string, repo string, env string, ipaddr string) error {
	result, err := db.Exec("DELETE FROM host_version WHERE owner=$1 AND repository=$2 AND environment=$3 AND ipaddr=$4", owner, repo, env, ipaddr)
	if err != nil {
		return err
	}

	num, err := result.RowsAffected()
	if num != 1 {
		return fmt.Errorf("Not found")
	}

	return err
}

func statusStr(alive bool) string {
	if alive {
		return HostStatusAlive
	}
	return HostStatusDown
}
