package email

import (
	"fmt"

	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ses"
)

var (
	defaultRegion  = ""
	defaultSender  = ""
	defaultCharSet = ""
)

// Email struct is an input
type Email struct {
	// optional fields for users to override
	Region  string
	Sender  string
	CharSet string

	// fields user must provide
	Recipients []string
	Subject    string
	Text       string
}

// Init sets default values
func Init(region string, sender string, charset string) {
	defaultRegion = region
	defaultSender = sender
	defaultCharSet = charset
}

// FillDefaults fills Email struct with default values
func (e *Email) fillDefaults() {
	if e.Region == "" {
		e.Region = defaultRegion
	}
	if e.Sender == "" {
		e.Sender = defaultSender
	}
	if e.CharSet == "" {
		e.CharSet = defaultCharSet
	}
}

// CheckInput validates input struct
func (e *Email) checkInput() error {
	if e.Region == "" || e.Sender == "" || e.CharSet == "" ||
		len(e.Recipients) == 0 || e.Subject == "" {
		return fmt.Errorf("Invalid email input: %+v", e)
	}
	return nil
}

// SendEmail sends an email via AWS SES
func SendEmail(e *Email) error {
	e.fillDefaults()
	err := e.checkInput()
	if err != nil {
		log.Errorf("%v", err)
		return err
	}

	sess, err := session.NewSession(&aws.Config{Region: aws.String(e.Region)})
	if err != nil {
		log.Errorf("%v", err)
		return err
	}

	input := &ses.SendEmailInput{
		Source: aws.String(e.Sender),
		Destination: &ses.Destination{
			ToAddresses: aws.StringSlice(e.Recipients),
		},
		Message: &ses.Message{
			Subject: &ses.Content{
				Charset: aws.String(e.CharSet),
				Data:    aws.String(e.Subject),
			},
			Body: &ses.Body{
				Text: &ses.Content{
					Charset: aws.String(e.CharSet),
					Data:    aws.String(e.Text),
				},
			},
		},
	}

	result, err := ses.New(sess).SendEmail(input)
	if err != nil {
		log.Errorf("Failed to send email: %v", err)
		return err
	}

	log.Infof("Email sent: %s, %+v", *result.MessageId, e)
	return nil
}
