package environment

import (
	"encoding/json"
	"net/http"

	"code.justin.tv/dta/skadi/pkg/config"
	"github.com/gorilla/mux"
	"github.com/jmoiron/sqlx"
)

var (
	db *DB
)

func RegisterHandlers(r *mux.Router, dbInstance *sqlx.DB) {
	db = &DB{
		dbInstance,
	}

	config.CreateHandler(r, "/v1/environments",
		ListHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")

	config.CreateHandler(r, "/v1/environments/{name:[A-Za-z0-9\\-_]+}",
		GetEnvironmentHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
}

const (
	defaultListPerPage = 10
)

type ListEnvironmentsOptions struct {
	PerPage int `schema:"per_page"`
	Page    int
}

func ListHandler(w http.ResponseWriter, r *http.Request) {
	options := &ListEnvironmentsOptions{}

	if err := config.ParseReq(r, options); err != nil {
		config.JSONError(w, http.StatusBadRequest, "", err)
		return
	}

	environments, err := db.ListEnvironments(options)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error executing list_environments", err)
		return
	}
	json.NewEncoder(w).Encode(environments)
}

func GetEnvironmentHandler(w http.ResponseWriter, r *http.Request) {
	name := mux.Vars(r)["name"]
	environment, err := db.GetEnvironment(name)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, err.Error(), err)
	} else if environment == nil {
		w.WriteHeader(http.StatusNotFound)
	} else {
		json.NewEncoder(w).Encode(environment)
	}
}
