package environment

import (
	"fmt"

	"github.com/jmoiron/sqlx"
	"github.com/lann/squirrel"
)

type DB struct {
	*sqlx.DB
}

// GetEnvrionment fetches and returns a Environment from the database.
func (db *DB) GetEnvironment(name string) (*Environment, error) {
	var e *Environment
	rows, err := db.Queryx("SELECT * FROM environments WHERE environment=$1", name)
	if err != nil {
		return nil, fmt.Errorf("Error querying db: %v", err)
	}
	defer rows.Close()

	services := make(map[string]ServiceArtifact)
	for rows.Next() {
		var s Service
		err := rows.StructScan(&s)
		if err != nil {
			return nil, err
		}
		var deployedSHA, deployedBranch, rollbackSHA, rollbackBranch string

		if s.DeployedSHA != nil {
			deployedSHA = *s.DeployedSHA
		}
		if s.DeployedBranch != nil {
			deployedBranch = *s.DeployedBranch
		}
		if s.RollbackSHA != nil {
			rollbackSHA = *s.RollbackSHA
		}
		if s.RollbackBranch != nil {
			rollbackBranch = *s.RollbackBranch
		}
		services[fmt.Sprintf("%s/%s", s.Owner, s.Repository)] = ServiceArtifact{Artifact{deployedSHA, deployedBranch}, Artifact{rollbackSHA, rollbackBranch}}
	}

	// Return nil if no services found (i.e. environment doesn't exist)
	if len(services) == 0 {
		return nil, nil
	}
	e = new(Environment)
	e.Name = name
	e.Services = services
	return e, nil
}

func (db *DB) ListEnvironments(options *ListEnvironmentsOptions) ([]*Environment, error) {
	// Set the request with default values:
	if options.Page < 1 {
		options.Page = 1
	}
	if options.PerPage == 0 {
		options.PerPage = defaultListPerPage
	}

	psql := squirrel.StatementBuilder.PlaceholderFormat(squirrel.Dollar)

	// hits a database which we can filter based on time, or repo/environment
	results := []*Environment{}

	// Build basic query:
	query := psql.Select("environment").Distinct().From("environments")

	query = query.OrderBy("environment asc").Limit(uint64(options.PerPage)).Offset(uint64(options.PerPage * (options.Page - 1)))
	q, args, err := query.ToSql()
	if err != nil {
		return nil, err
	}

	err = db.Select(&results, q, args...)
	if err != nil {
		return nil, fmt.Errorf("Error selecting rows: %v", err)
	}

	return results, nil
}
