package freeze

import (
	"fmt"
	"time"

	"github.com/jmoiron/sqlx"
	"github.com/lann/squirrel"
)

type DB struct {
	*sqlx.DB
}

type ListFreezeOptions struct {
	Owner       *string
	Repository  *string `schema:"repo"`
	Environment *string
	AfterDate   *time.Time
}

func (db *DB) InsertFreeze(f *Freeze) error {
	_, err := db.Exec("INSERT INTO freezes (owner, repository, creator, reason, environment) VALUES ($1, $2, $3, $4, $5)",
		f.Owner, f.Repo, f.Creator, f.Reason, f.Environment)
	return err
}

func (db *DB) GetFreeze(owner *string, repo *string) *Freeze {
	var f Freeze
	row := db.QueryRow("SELECT owner, repository, creator, reason, environment FROM freezes WHERE owner=$1 AND repository=$2 ORDER BY id DESC limit 1", owner, repo)
	err := row.Scan(&f.Owner, &f.Repo, &f.Creator, &f.Reason, &f.Environment)
	if err != nil {
		return nil
	} else {
		f.Freeze()
		return &f
	}
}

func (db *DB) ListFreezes(owner *string, repo *string) (*[]Freeze, error) {
	freezes := []Freeze{}
	rows, err := db.Query("SELECT ID, owner, repository, creator, reason, environment, createdat FROM freezes WHERE owner=$1 AND repository=$2 AND deleted is not true", owner, repo)
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	for rows.Next() {
		var f Freeze
		err := rows.Scan(&f.ID, &f.Owner, &f.Repo, &f.Creator, &f.Reason, &f.Environment, &f.CreatedAt)
		if err != nil {
			return nil, err
		}
		f.Freeze()
		freezes = append(freezes, f)
	}
	if err := rows.Err(); err != nil {
		return nil, err
	}
	return &freezes, nil
}

func (db *DB) GetFreezes(options *ListFreezeOptions) ([]*Freeze, error) {
	freezes := []*Freeze{}
	psql := squirrel.StatementBuilder.PlaceholderFormat(squirrel.Dollar)

	// Build basic query:
	query := psql.Select("*").From("freezes")

	// Dynamically build the rest of the query.
	for k, v := range map[string]*string{
		"owner":       options.Owner,
		"repository":  options.Repository,
		"environment": options.Environment,
	} {
		if v != nil {
			query = query.Where(fmt.Sprintf("%s = ?", k), *v)
		}
	}

	// only get completed freezes
	query = query.Where("deleted = true")

	if options.AfterDate != nil {
		query = query.Where("(createdat >= ? OR deletedat >= ?)", *options.AfterDate, *options.AfterDate)
	}

	q, args, err := query.ToSql()
	if err != nil {
		return nil, err
	}

	rows, err := db.Queryx(q, args...)
	if err != nil {
		return nil, fmt.Errorf("Error querying db: %v", err)
	}
	defer rows.Close()

	for rows.Next() {
		var f Freeze
		err := rows.StructScan(&f)
		if err != nil {
			return nil, err
		}
		f.Freeze()
		freezes = append(freezes, &f)
	}

	return freezes, nil
}

func (db *DB) DeleteFreeze(owner *string, repo *string) error {
	_, err := db.Exec("DELETE FROM freezes WHERE owner=$1 AND repository=$2", owner, repo)
	return err
}

func (db *DB) UnfreezeFreeze(id int, deleter *string) error {
	rows, err := db.Query("UPDATE freezes SET deleted=true, deletedby=$1, deletedat=CURRENT_TIMESTAMP WHERE id=$2", deleter, id)
	if err != nil {
		return err
	}
	defer rows.Close()
	return err
}
