package freeze

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"github.com/google/go-github/github"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/jmoiron/sqlx"
	"github.com/nlopes/slack"
)

// Freeze freezes an environment for a specific owner/repo.
type Freeze struct {
	ID          int        `json:"id" db:"id"`
	Owner       *string    `json:"owner" db:"owner"`
	Repo        *string    `json:"repo" db:"repository"`
	Creator     *string    `json:"creator" db:"creator"`
	Reason      *string    `json:"reason" db:"reason"`
	Environment *string    `json:"environment" db:"environment"`
	Deleted     bool       `json:"deleted" db:"deleted"`
	DeletedBy   *string    `json:"deleted_by" db:"deletedby"`
	DeletedAt   *time.Time `json:"deleted_at" db:"deletedat"`
	CreatedAt   *time.Time `json:"created_at" db:"createdat"`
	Type        string     `json:"type,omitempty"`
	IsFrozen    bool       `json:"is_frozen"`
	AutoDeploy  bool       `json:"autodeploy" db:"autodeploy"`

	Schedule
}

// Validate determines if a freeze is valid. If invalid, an error is returned
// describing why.
func (f Freeze) Validate() error {
	if f.Owner == nil {
		return errors.New("owner is required")
	} else if f.Repo == nil {
		return errors.New("repo is required")
	} else if f.Creator == nil {
		return errors.New("creator is required")
	} else if f.Reason == nil {
		return errors.New("reason is required")
	} else if f.Environment == nil {
		return errors.New("environment is required")
	}

	switch f.Type {
	case "schedule":
		if err := f.Schedule.Validate(); err != nil {
			return err
		}
	}

	return nil
}

func (f *Freeze) Freeze() {
	f.IsFrozen = f.Frozen()
}

func (f Freeze) Frozen() bool {
	switch f.Type {
	case "schedule":
		now := time.Now().In(timezone)

		ok, err := f.Schedule.FrozenAt(now)
		if err != nil {
			log.Errorf("unable to determine if schedule %d is frozen: %s", f.ID, err)
		}
		return ok
	default:

	}
	return true
}

type FreezeConfig struct {
	SlackClient  *slack.Client
	GithubClient *github.Client
	ConsulClient *consulapi.Client
	BaseURL      *url.URL
}

var (
	db         *DB
	fc         *FreezeConfig
	scheduleDB *ScheduleDB
)

func RegisterHandlers(r *mux.Router, dbInstance *sqlx.DB, freezeConfig *FreezeConfig) {
	db = &DB{dbInstance}
	fc = freezeConfig
	scheduleDB = &ScheduleDB{
		db: dbInstance,
	}

	routeOpts := &config.RouteOptions{AddCORS: true}

	// Original freezes
	config.CreateHandler(r, "/v1/freeze/create",
		createFreezeHandler,
		routeOpts,
	).Methods("POST")
	config.CreateHandler(r, "/v1/freeze/get",
		getFreezeHandler,
		routeOpts,
	).Methods("GET")
	config.CreateHandler(r, "/v1/freeze/delete",
		deleteFreezeHandler,
		routeOpts,
	).Methods("POST")
	config.CreateHandler(r, "/v1/freeze/list",
		listFreezesHandler,
		routeOpts,
	)
	config.CreateHandler(r, "/v1/freeze/unfreeze",
		unfreezeFreezeHandler,
		routeOpts,
	).Methods("POST")
}

func deleteFreezeHandler(w http.ResponseWriter, r *http.Request) {
	decoder := json.NewDecoder(r.Body)
	var f Freeze
	err := decoder.Decode(&f)
	if err != nil {
		config.JSONError(w, 500, "deleteFreezeHandler unable to decode JSON", err)
	}
	owner := f.Owner
	repo := f.Repo

	switch f.Type {
	case "schedule":
		config.JSONError(w, http.StatusBadRequest, "cannot delete scheduled freezes, please unfreeze scheduled freezes", err)
		return
	default:
		log.Printf("Deleting a freeze: %v/%v", *owner, *repo)
		err = db.DeleteFreeze(owner, repo)
		if err != nil {
			config.JSONError(w, 500, "error deleting freeze", err)
		}
	}

	w.Write([]byte("ok"))
}

func getFreezeHandler(w http.ResponseWriter, r *http.Request) {
	var f *Freeze
	scheduled := r.FormValue("scheduled")
	if scheduled != "" {
		sID := r.FormValue("id")
		id, err := strconv.Atoi(sID)
		if err != nil {
			config.JSONError(w, http.StatusBadRequest, "unable to parse id", err)
			return
		}

		f, err = scheduleDB.GetScheduleFreeze(id)
		if err != nil && !notFoundError(err) {
			config.JSONError(w, http.StatusInternalServerError, "unable to get scheduled freeze", err)
			return
		}
	} else {
		owner := r.FormValue("owner")
		repo := r.FormValue("repo")

		f = db.GetFreeze(&owner, &repo)
	}

	b, err := json.Marshal(f)
	if err != nil {
		config.JSONError(w, 500, "error getting freeze", err)
		return
	}
	w.Write(b)
}

func listFreezes(owner, repo string) (*[]Freeze, error) {
	f, err := db.ListFreezes(&owner, &repo)
	if err != nil {
		log.Error("error getting freezes from db:", err)
		return nil, err
	}

	scheduled, err := scheduleDB.ListScheduleFreezesByOR(owner, repo)
	if err != nil {
		log.Error("error listing scheduled freezes:", err)
		return nil, err
	}

	*f = append(*f, *scheduled...)

	return f, nil
}

func listFreezesHandler(w http.ResponseWriter, r *http.Request) {
	owner := r.FormValue("owner")
	repo := r.FormValue("repo")

	f, err := listFreezes(owner, repo)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error listing freezes", err)
		return
	}

	b, err := json.Marshal(f)
	if err != nil {
		config.JSONError(w, 500, "error getting freezes", err)
	}
	w.Write(b)
}

func IsFrozen(owner, repo, env string) (bool, error) {
	freezes, err := listFreezes(owner, repo)
	if err != nil {
		return false, err
	}
	for _, freeze := range *freezes {
		if *freeze.Environment == env {
			if freeze.Frozen() {
				return true, nil
			}
		}
	}
	return false, nil
}

func unfreezeFreezeHandler(w http.ResponseWriter, r *http.Request) {
	decoder := json.NewDecoder(r.Body)
	var f Freeze
	err := decoder.Decode(&f)
	if err != nil {
		config.JSONError(w, 500, "unfreezeFreezeHandler unable to decode JSON", err)
	}
	id := f.ID
	deleter := f.DeletedBy

	switch f.Type {
	case "schedule":
		if err := scheduleDB.DeleteScheduleFreeze(id, *deleter); err != nil {
			config.JSONError(w, 500, "error unfreezing freeze", err)
			return
		}
	default:
		log.Printf("Unfreezing a freeze - id: %v, by: %v", id, *deleter)
		err = db.UnfreezeFreeze(id, deleter)
		if err != nil {
			config.JSONError(w, 500, "error unfreezing freeze", err)
			return
		}
	}

	action := "unfreeze"
	reason := "null"
	err = broadcastFreezeEvent(*f.Owner, *f.Repo, *f.DeletedBy, *f.Environment, action, reason)
	if err != nil {
		log.Printf("Could not broadcast freeze for %v/%v because %v", *f.Owner, *f.Repo, err)
	}
	w.Write([]byte("ok"))
}

func createFreezeHandler(w http.ResponseWriter, r *http.Request) {
	log.Printf("Creating a new freeze: %v", &r.Body)
	decoder := json.NewDecoder(r.Body)
	defer r.Body.Close()
	var f Freeze
	err := decoder.Decode(&f)
	if err != nil {
		config.JSONError(w, 500, "createFreezeHandler unable to decode JSON", err)
		return
	}

	switch f.Type {
	case "schedule":
		if err := f.Validate(); err != nil {
			config.JSONError(w, http.StatusBadRequest, "freeze failed validation", err)
			return
		}

		if err := scheduleDB.InsertScheduleFreeze(&f); err != nil {
			config.JSONError(w, http.StatusInternalServerError, "error inserting freeze", err)
			return
		}
	default:
		if f.Owner == nil {
			config.JSONError(w, 500, "New freeze has no owner", err)
			return
		} else if f.Repo == nil {
			config.JSONError(w, 500, "New freeze has no repo", err)
			return
		} else if f.Creator == nil {
			config.JSONError(w, 500, "New freeze has no creator", err)
			return
		} else if f.Reason == nil {
			config.JSONError(w, 500, "New freeze has no reason", err)
			return
		} else if f.Environment == nil {
			config.JSONError(w, 500, "New freeze has no environment", err)
			return
		}

		err = db.InsertFreeze(&f)
		if err != nil {
			config.JSONError(w, 500, "Error inserting a freeze", err)
			return
		}
	}

	action := "create"
	err = broadcastFreezeEvent(*f.Owner, *f.Repo, *f.Creator, *f.Environment, action, *f.Reason)
	if err != nil {
		log.Printf("Could not broadcast freeze for %v/%v because %v", *f.Owner, *f.Repo, err)
	}
	w.Write([]byte("ok"))
}

func broadcastFreezeEvent(owner, repo, user, env, action, reason string) error {
	var color, title, text string

	chatConfig, err := config.LoadChatConfig(fc.GithubClient, fc.ConsulClient, owner, repo, githubcache.GetDefaultBranch(owner, repo))
	if err != nil || chatConfig == nil {
		return fmt.Errorf("error loading chat config: %v", err)
	}
	baseURL := fc.BaseURL.String()

	fields := []slack.AttachmentField{}

	switch action {
	case "create":
		color = "danger"
		title = "Freeze Created"
		text = fmt.Sprintf("%v froze %v on %v/%v", user, env, owner, repo)
		f := slack.AttachmentField{
			Title: "Reason",
			Value: fmt.Sprintf("%v", reason),
			Short: false,
		}
		fields = append(fields, f)
	case "unfreeze":
		color = "warning"
		title = "Freeze Removed"
		text = fmt.Sprintf("%v unfroze %v on %v/%v", user, env, owner, repo)
	}

	gheBaseURL := fc.GithubClient.BaseURL.String()
	gheBaseURL = strings.Replace(gheBaseURL, "/api/v3/", "/", -1)
	if chatConfig.Channel != nil {
		_, _, err = fc.SlackClient.PostMessage(*chatConfig.Channel, "", slack.PostMessageParameters{
			Username: "skadi",
			IconURL:  "http://s.jtvnw.net/jtv_user_pictures/hosted_images/GlitchIcon_PurpleonWhite.png",
			Attachments: []slack.Attachment{
				slack.Attachment{
					Color:      color,
					AuthorName: user,
					AuthorLink: fmt.Sprintf("%v%v", gheBaseURL, user),
					Title:      title,
					TitleLink:  fmt.Sprintf("%v#/%v/%v", baseURL, owner, repo),
					Text:       text,
					Fields:     fields,
				},
			},
		})
		if err != nil {
			return err
		}
	}

	return nil
}

func GetFreezes(options *ListFreezeOptions) ([]*Freeze, error) {
	results, err := db.GetFreezes(options)
	if err != nil {
		return nil, err
	}

	return results, nil
}
