package freeze

import (
	"errors"
	"time"

	log "github.com/Sirupsen/logrus"
	"github.com/jmoiron/sqlx/types"
)

var (
	timezone *time.Location
)

func init() {
	var err error
	timezone, err = time.LoadLocation("US/Pacific")
	if err != nil {
		log.Fatal("failed to load pacific time zone")
	}
}

// ScheduleFreeze is a freeze on a schedule. The schedule is determined by
// a start date, end date, and an optional expression. The expression is
// used to express data like specific days and times to apply the freeze.
type Schedule struct {
	StartDate      *time.Time     `json:"start_date,omitempty"`
	EndDate        *time.Time     `json:"end_date,omitempty"`
	ExpressionType string         `json:"expression_type,omitempty"`
	Expression     types.JSONText `json:"expression,omitempty"`
}

// Localize locals the times in schedule to a location.
func (s *Schedule) Localize() {
	if s.StartDate != nil {
		*s.StartDate = s.StartDate.In(timezone)
	}
	if s.EndDate != nil {
		*s.EndDate = s.EndDate.In(timezone)
	}
}

// Freezer determines if a schedule is frozen at a particular time.
type Freezer interface {
	FrozenAt(time.Time) (bool, error)
}

// Validate determines if a schedule freeze is valid. If invalid, an error is returned
// describing why.
func (f *Schedule) Validate() error {
	if f.EndDate != nil && f.StartDate == nil {
		return errors.New("start date is required when end date is provided")
	}

	f.Localize()

	return nil
}

// FrozenAt returns whether or not this schedule is frozen for a given day
func (f Schedule) FrozenAt(now time.Time) (bool, error) {
	// First ensure that this day is in our range.
	inRange := true
	if f.StartDate != nil {
		inRange = f.StartDate.Before(now)
	}
	if f.EndDate != nil {
		inRange = inRange && f.EndDate.After(now)
	}
	if !inRange {
		return false, nil
	}

	// Evaluate the expression, if we support the expression type.
	switch f.ExpressionType {
	case "weekly":
		weekly, err := NewWeeklySchedule(f.Expression)
		if err != nil {
			return false, err
		}

		return weekly.FrozenAt(now)
	case "weekly-repeat":
		weekly, err := NewWeeklyRepeatSchedule(f.Expression)
		if err != nil {
			return false, err
		}

		return weekly.FrozenAt(now)
	default:
		// Do nothing
	}

	return true, nil
}
