package freeze

import (
	"encoding/json"
	"time"
)

var (
	weekdaysByIndex = map[time.Weekday]int{
		time.Sunday:    0,
		time.Monday:    1,
		time.Tuesday:   2,
		time.Wednesday: 3,
		time.Thursday:  4,
		time.Friday:    5,
		time.Saturday:  6,
	}
)

type WeeklyRepeatSchedule struct {
	StartDay  int       `json:"start_day"`
	StartTime time.Time `json:"start_time"`
	EndDay    int       `json:"end_day"`
	EndTime   time.Time `json:"end_time"`
}

func NewWeeklyRepeatSchedule(b []byte) (*WeeklyRepeatSchedule, error) {
	var weekly WeeklyRepeatSchedule
	err := json.Unmarshal(b, &weekly)
	if err != nil {
		return nil, err
	}

	weekly.StartTime = weekly.StartTime.In(timezone)
	weekly.EndTime = weekly.EndTime.In(timezone)

	return &weekly, nil
}

func (s WeeklyRepeatSchedule) FrozenAt(now time.Time) (bool, error) {
	day := now.Weekday()
	dayNum := weekdaysByIndex[day]
	hour := now.Hour()
	minute := now.Minute()

	// In all cases, verify times if day equals start day
	if dayNum == s.StartDay {
		// Make sure time is after start time
		sHour := s.StartTime.Hour()
		sMin := s.StartTime.Minute()
		// Current hour must be greater than or equal to schedule hour
		if hour < sHour {
			return false, nil
		} else if hour == sHour {
			// If equal, current minute is greater than or equal to schedule minute
			if minute < sMin {
				return false, nil
			}
		}
	}

	// In all cases, verify times if day equals end day
	if dayNum == s.EndDay {
		// Make sure time is before end time
		sHour := s.EndTime.Hour()
		sMin := s.EndTime.Minute()
		// Current hour must be less than or equal to schedule hour
		if hour > sHour {
			return false, nil
		} else if hour == sHour {
			// If equal, current minute is less than or equal to schedule minute
			if minute > sMin {
				return false, nil
			}
		}
	}

	// Are we in a situation where the start day is 5 (Friday) and the end day is 1 (Monday)
	if s.EndDay < s.StartDay {
		// Current time must be greater than startday but less than end day
		if dayNum >= s.StartDay && dayNum < 7 {
			// start time already verified
			return true, nil
		} else if dayNum <= s.EndDay {
			// end time already verified
			return true, nil
		} else {
			return false, nil
		}
		// Normal case where start date is less than or equal to end date
	} else if dayNum < s.StartDay || dayNum > s.EndDay {
		return false, nil
	}

	return true, nil
}
