package freeze

import (
	"os"
	"testing"
	"time"
)

type weeklyRepeatScheduleCase struct {
	Expression string
	Now        time.Time
	Valid      bool
}

func TestWeeklyRepeatScheduleFrozenAt(t *testing.T) {
	cases := map[string]weeklyRepeatScheduleCase{
		"friday to monday, valid (saturday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day": 5, "end_day": 1, "start_time": "1899-12-31T08:00:00.000Z", "end_time": "1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("25 Jun 16 15:04 -0700"),
			Valid:      true,
		},
		"friday to monday, valid (monday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day": 5, "end_day": 1, "start_time": "1899-12-31T08:00:00.000Z", "end_time": "1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("27 Jun 16 8:04 -0700"),
			Valid:      false,
		},
		"friday to monday, invalid (monday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day": 5, "end_day": 1, "start_time": "1899-12-31T08:00:00.000Z", "end_time": "1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("27 Jun 16 15:04 -0700"),
			Valid:      false,
		},
		"friday to monday, invalid (tuesday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day": 5, "end_day": 1, "start_time": "1899-12-31T08:00:00.000Z", "end_time": "1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("21 Jun 16 15:04 -0700"),
			Valid:      false,
		},
		"monday to thursday, valid (tuesday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day": 1, "end_day": 4, "start_time": "1899-12-31T08:00:00.000Z", "end_time": "1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("21 Jun 16 15:04 -0700"),
			Valid:      true,
		},
		"monday to thursday, invalid (saturday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day": 1, "end_day": 4, "start_time": "1899-12-31T08:00:00.000Z", "end_time": "1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("25 Jun 16 15:04 -0700"),
			Valid:      false,
		},
		"friday to tuesday, invalid (wednesday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day":5,"end_day":2,"start_time":"1899-12-31T08:00:00.000Z","end_time":"1899-12-31T08:00:00.000Z"}`,
			Now:        mustParse("22 Jun 16 15:04 -0700"),
			Valid:      false,
		},
		"thursday 12 PM to 5 PM, valid (thursday)": weeklyRepeatScheduleCase{
			Expression: `{"start_day":4,"end_day":4,"start_time":"1899-12-31T20:00:00.000Z","end_time":"1900-01-01T01:00:00.000Z"}`,
			Now:        mustParse("07 Jul 16 15:04 -0700"),
			Valid:      true,
		},
	}

	specificCase := os.Getenv("SPECIFIC_TEST_CASE")

	for name, c := range cases {
		if specificCase != "" && name != specificCase {
			continue
		}

		schedule, err := NewWeeklyRepeatSchedule([]byte(c.Expression))
		if err != nil {
			t.Errorf("%s: failed to create weekly repeat schedule: %s", name, err)
			continue
		}

		ok, err := schedule.FrozenAt(c.Now)
		if ok == c.Valid {
			if c.Valid && err != nil {
				t.Errorf("%s: encountered error: %s", name, err)
				continue
			}
			// if ok and c.Valid match, we were right
			continue
		}

		t.Errorf("%s: expected %t, got %t. err: %s", name, c.Valid, ok, err)
	}
}
