package freeze

import (
	"testing"
	"time"
)

type weeklyScheduleCase struct {
	Expression string
	Now        time.Time
	Valid      bool
}

func mustParse(format string) time.Time {
	t, err := time.Parse(time.RFC822Z, format)
	if err != nil {
		panic(err)
	}

	return t
}

func TestWeeklyScheduleFrozenAt(t *testing.T) {
	cases := map[string]weeklyScheduleCase{
		"same day, only day": weeklyScheduleCase{
			Expression: `{"days": {"Su": true}}`,
			Now:        mustParse("12 Jun 16 15:04 -0700"),
			Valid:      true,
		},
		"multiple days": weeklyScheduleCase{
			Expression: `{"days": {"Su": true, "Sa": true}}`,
			Now:        mustParse("11 Jun 16 15:04 -0700"),
			Valid:      true,
		},
		"all but one": weeklyScheduleCase{
			Expression: `{"days": {"Su": true, "M": true, "Tu": true, "W": true, "Th": true, "F": true}}`,
			Now:        mustParse("11 Jun 16 15:04 -0700"),
		},
	}

	for name, c := range cases {
		schedule, err := NewWeeklySchedule([]byte(c.Expression))
		if err != nil {
			t.Errorf("%s: failed to create weekly schedule: %s", name, err)
			continue
		}

		ok, err := schedule.FrozenAt(c.Now)
		if ok == c.Valid {
			if c.Valid && err != nil {
				t.Errorf("%s: encountered error: %s", name, err)
				continue
			}
			// if ok and c.Valid match, we were right
			continue
		}

		t.Errorf("%s: expected %t, got %t. err: %s", name, c.Valid, ok, err)
	}
}
