// Package github provides an extension to the google golang github package
// that wraps API calls with interface client parameters for
// testability.
package github

import (
	"context"
	"errors"

	gh "github.com/google/go-github/github"
)

// Client is an extension of the github.com/google/go-github/github.Client.
type Client struct {
	*gh.Client
}

// GetUser is a client interface that provides the get user call
// (https://developer.github.com/v3/users/#get-a-single-user).
type GetUser interface {
	GetUser(string) (*gh.User, error)
}

// GetUser calls the GitHub API endpoint to get a single user.
func (client *Client) GetUser(login string) (*gh.User, error) {
	user, _, err := client.Users.Get(context.TODO(), login)
	if err != nil {
		return nil, err
	}
	return user, nil
}

// ListDeployments is a client interface that provides the list deployments
// call
// (https://godoc.org/github.com/google/go-github/github#RepositoriesService.ListDeployments)
type ListDeployments interface {
	ListDeployments(owner, repo string, opt *gh.DeploymentsListOptions) ([]*gh.Deployment, error)
}

// ListDeployments calls the GitHub API endpoint to list deployments and returns the results
func (client *Client) ListDeployments(owner, repo string, opt *gh.DeploymentsListOptions) ([]*gh.Deployment, error) {
	deployments, _, err := client.Repositories.ListDeployments(context.TODO(), owner, repo, opt)
	if err != nil {
		return nil, err
	}
	if len(deployments) == 0 {
		return nil, errors.New("deployment statuses is empty")
	}
	return deployments, nil
}

// GetFirstDeploymentStatus is a client interface that provides a convenience
// function that calls list deployment statuses
// (https://developer.github.com/v3/repos/deployments/#list-deployment-statuses)
// and returns the first status.
type GetFirstDeploymentStatus interface {
	GetFirstDeploymentStatus(owner, repo string, deployment int) (*gh.DeploymentStatus, error)
}

// GetFirstDeploymentStatus calls the GitHub API endpoint to list status
// updates for a deployment and returns the first update.
func (client *Client) GetFirstDeploymentStatus(owner, repo string, deployment int64) (*gh.DeploymentStatus, error) {
	// opt := &gh.ListOptions{}
	opt := &gh.ListOptions{PerPage: 1}
	statuses, _, err := client.Repositories.ListDeploymentStatuses(context.TODO(), owner, repo, deployment, opt)
	if err != nil {
		return nil, err
	}
	if len(statuses) == 0 {
		return nil, errors.New("deployment statuses is empty")
	}
	return statuses[0], nil
}

// CompareRepositoryCommits is a client interface that provides the repoistory
// compare commits call
// (https://developer.github.com/v3/repos/commits/#compare-two-commits).
type CompareRepositoryCommits interface {
	CompareRepositoryCommits(owner, repo, previousSHA, SHA string) (*gh.CommitsComparison, error)
}

// CompareRepositoryCommits calls the GitHub API endpoint to compare two
// commits in a repository.
func (client *Client) CompareRepositoryCommits(owner, repo, previousSHA, SHA string) (*gh.CommitsComparison, error) {
	commitsComparison, _, err := client.Repositories.CompareCommits(context.TODO(), owner, repo, previousSHA, SHA)
	if err != nil {
		return nil, err
	}
	return commitsComparison, nil
}
