package githubcache

import (
	"fmt"
	"github.com/google/go-github/github"
	"github.com/pmylund/go-cache"
	"golang.org/x/net/context"
	"time"
)

const (
	repositoryInfoCacheExpiration      = 10 * time.Minute
	repositoryInfoCacheCleanupInterval = 1 * time.Minute
)

var (
	initialized         bool
	githubClient        *github.Client
	repositoryInfoCache *cache.Cache
)

// Init sets global github client
func Init(client *github.Client) {
	if initialized {
		return
	}
	githubClient = client
	repositoryInfoCache = cache.New(repositoryInfoCacheExpiration, repositoryInfoCacheCleanupInterval)
	initialized = true
}

// GetRepositoryInfo returns github repository structure.
// It looks up from the cache then queires to git if not found in cache.
func GetRepositoryInfo(owner string, repo string) *github.Repository {
	if !initialized {
		return nil
	}

	key := generateRepositoryCacheKey(owner, repo)
	var r *github.Repository
	val, _ := repositoryInfoCache.Get(key)
	if val != nil {
		r = val.(*github.Repository)
	} else {
		r, _, _ = githubClient.Repositories.Get(context.TODO(), owner, repo)
		if r != nil {
			repositoryInfoCache.Set(key, r, 0)
		}
	}
	return r
}

// GetDefaultBranch returns the default branch name.
// It looks up from the cache then queires to git if not found in cache.
// If the repository does not exist it will return "master" as it'll be the default branch upon the creation.
func GetDefaultBranch(owner string, repo string) string {
	r := GetRepositoryInfo(owner, repo)
	if r == nil || r.DefaultBranch == nil {
		return "master"
	}
	return r.GetDefaultBranch()
}

func generateRepositoryCacheKey(owner string, repo string) string {
	return fmt.Sprintf("%s/%s", owner, repo)
}

func getGitRepository(githubClient *github.Client, owner string, repo string) (*github.Repository, error) {
	repository, _, err := githubClient.Repositories.Get(context.TODO(), owner, repo)
	return repository, err
}
