package helpers

// Fail-Safe Consul API

import (
	"fmt"
	"strings"
	"sync"

	"code.justin.tv/common/chitin"
	twapi "code.justin.tv/release/twitch-consul-api"
	log "github.com/Sirupsen/logrus"
	consulapi "github.com/hashicorp/consul/api"
	"golang.org/x/net/context"
)

type FSConsulClient struct {
	client      *consulapi.Client // default client
	clientMap   map[string]*consulapi.Client // client pool for different consul servers
}

func NewFailSafeConsulClient(client *consulapi.Client) *FSConsulClient {
	return &FSConsulClient{
		client:      client,
		clientMap:   make(map[string]*consulapi.Client),
	}
}

func (c *FSConsulClient) GetConsulClient(consulHost string) (*consulapi.Client, error) {
	if consulHost == "" {
		return c.client, nil
	}
	client, ok := c.clientMap[consulHost]
	if ok {
		return client, nil
	}

	client, err := twapi.NewClient(consulHost, chitin.Client(context.Background()))
	if err != nil {
		return nil, err
	}
	c.clientMap[consulHost] = client
	return client, nil
}

//func (c *FSConsulClient) GetDatacenters() []string {
//	return c.datacenters
//}

func (c *FSConsulClient) ListDatacenters(datacenters []string) (map[string]*twapi.Datacenter, error) {
	var wait sync.WaitGroup
	dcs := map[string]*twapi.Datacenter{}
	datacenters = UniqueStringsSorted(datacenters)
	dcErrorChan := make(chan string, len(datacenters))
	for _, datacenter := range datacenters {
		consulHost, dcName := ParseDcName(datacenter)
		consulClient, err := c.GetConsulClient(consulHost)
		if err != nil {
			log.Errorf("Error getting consul client for %v : %v", consulHost, err)
			continue
		}
		dc, err := twapi.NewDatacenter(dcName, consulClient)
		// Adjust the name after setting up a datacenter structure
		dc.Name = datacenter

		if err != nil {
			// By the api, this never happens
			log.Errorf("Error setting up datacenter %q: %v", dc.Name, err)
			continue
		}

		wait.Add(1)
		go func(localDC *twapi.Datacenter) {
			defer wait.Done()
			err := Retry(func() error {
				return localDC.GetData()
			})
			if err != nil {
				log.Errorf("Error fetching data for datacenter %q: %v", localDC.Name, err)
				dcErrorChan <- localDC.Name
				return
			}
		}(dc)
		dcs[dc.Name] = dc
	}
	wait.Wait()
	close(dcErrorChan)

	// Merge the DCs that have failed to look up
	dcsError := SerializeStringChan(&dcErrorChan)

	if len(dcsError) > 0 {
		return nil, fmt.Errorf("Unable to reach datacenters: %v", dcsError)
	}

	return dcs, nil
}

func (c *FSConsulClient) GetAliveHostnamesWithDCs(services []string, tag string, datacenters []string) ([]twapi.DCtoNodes, error) {
	if services == nil {
		return nil, fmt.Errorf("GetAliveHostnamesWithDCs: Invalid argument.")
	}

	datacenters = UniqueStringsSorted(datacenters)
	dcs, err := c.ListDatacenters(datacenters)
	if err != nil {
		return nil, err
	}

	nodesDcMap := make(map[string][]string)
	for _, service := range services {
		dcsToNodes, dcsError := mapLookupServiceWithDCs(dcs, service, tag)
		if len(dcsError) > 0 {
			return nil, fmt.Errorf("Unable to lookup from datacenters: %v", dcsError)
		}

		for _, dcToNodes := range dcsToNodes {
			if _, ok := nodesDcMap[dcToNodes.DC]; ok {
				nodesDcMap[dcToNodes.DC] = append(nodesDcMap[dcToNodes.DC], dcToNodes.Nodes...)
			} else {
				nodesDcMap[dcToNodes.DC] = dcToNodes.Nodes
			}
		}
	}

	dcToNodesList := []twapi.DCtoNodes{}
	for dc, nodes := range nodesDcMap {
		//Unique is called because multiple services can have the same hostname
		dcToNodesList = append(dcToNodesList, []twapi.DCtoNodes{twapi.DCtoNodes{DC: dc, Nodes: UniqueStrings(nodes)}}...)
	}

	// Find datacenters that have no target hosts and add them to the list with empty target hosts.
	for _, dcName := range datacenters {
		if _, ok := nodesDcMap[dcName]; !ok {
			dcToNodesList = append(dcToNodesList, []twapi.DCtoNodes{twapi.DCtoNodes{DC: dcName, Nodes: []string{}}}...)
		}
	}

	return dcToNodesList, nil
}

func (c *FSConsulClient) UpdateKV(key string, value string, datacenter string) error {
	consulHost, dcName := ParseDcName(datacenter)
	consulClient, err := c.GetConsulClient(consulHost)
	if err != nil {
		return err
	}
	_, err = consulClient.KV().Put(
		&consulapi.KVPair{Key: key, Value: []byte(value)},
		&consulapi.WriteOptions{Datacenter: dcName})
	return err
}

func mapLookupServiceWithDCs(dcs map[string]*twapi.Datacenter, service string, tag string) ([]twapi.DCtoNodes, []string) {
	var wait sync.WaitGroup

	dcsChan := make(chan twapi.DCtoNodes, len(dcs))
	dcsErrorChan := make(chan string, len(dcs))
	for _, dc := range dcs {
		wait.Add(1)
		go func(localDC *twapi.Datacenter) {
			defer wait.Done()
			var hosts []string
			err := Retry(func() error {
				var err error
				hosts, err = localDC.LookupService(service, tag, true)
				return err
			})
			if err != nil {
				log.Errorf("Error looking up service %q in datacenter %q: %v", service, localDC.Name, err)
				dcsErrorChan <- localDC.Name
				return
			}
			if len(hosts) > 0 {
				dcsChan <- twapi.DCtoNodes{DC: localDC.Name, Nodes: hosts}
			}
		}(dc)
	}
	wait.Wait()
	close(dcsChan)
	close(dcsErrorChan)

	// Merge the hosts:
	dcsToNodes := []twapi.DCtoNodes{}
	for dcToNodes := range dcsChan {
		dcsToNodes = append(dcsToNodes, []twapi.DCtoNodes{dcToNodes}...)
	}

	return dcsToNodes, SerializeStringChan(&dcsErrorChan)
}

func ParseDcName(dcName string) (string, string) {
	s := strings.Split(dcName, "/")
	if len(s) < 2 {
		return "", s[0]
	}
	return s[0], s[1]
}

func GetHostnamesFromDCtoNodes(dcsToNodes []twapi.DCtoNodes) []string {
	var allNodes []string
	for _, dcToNode := range dcsToNodes {
		allNodes = append(allNodes, dcToNode.Nodes...)
	}
	return UniqueStrings(allNodes)
}

func GetDatacenterNamesFromDCtoNodes(dcsToNodes []twapi.DCtoNodes) []string {
	var allDcs []string
	for _, dcToNode := range dcsToNodes {
		allDcs = append(allDcs, []string{dcToNode.DC}...)
	}
	return allDcs
}
