package helpers

import (
	"code.justin.tv/common/chitin"
	twapi "code.justin.tv/release/twitch-consul-api"
	"fmt"
	"golang.org/x/net/context"
	"sort"
)

//
// These are actually tests. Go matches the output of the logic with "Output:" sections comments.
// If output doesn't match, it'll report an error. It's just a lot easier doing these tests in this way.
//

const (
	consulEndpoint = "api.us-west-2.prod.consul.live-video.a2z.com"
)

func Example_init() {
	// Change retry mode to non-time based.
	Retries = 1
}

func Example_listDatacenters() {
	cClient, _ := twapi.NewClient(consulEndpoint, chitin.Client(context.Background()))
	fsConsul := NewFailSafeConsulClient(cClient)
	dcs, _ := fsConsul.ListDatacenters(SplitStringNoEmpty("wanmaster", ","))
	fmt.Println(dcs["wanmaster"] != nil)
	fmt.Println(dcs["us-west2"] != nil)
	// Output:
	// true
	// false
}

func Example_listMultipleDatacenters() {
	cClient, _ := twapi.NewClient(consulEndpoint, chitin.Client(context.Background()))
	fsConsul := NewFailSafeConsulClient(cClient)
	dcs, _ := fsConsul.ListDatacenters(SplitStringNoEmpty(fmt.Sprintf("wanmaster, %s/wanmaster", consulEndpoint), ","))
	fmt.Println(dcs["wanmaster"] != nil)
	fmt.Println(dcs[fmt.Sprintf("%s/wanmaster", consulEndpoint)] != nil)
	fmt.Println(dcs["us-west2"] != nil)
	// Output:
	// true
	// true
	// false
}

func Example_getHostnamesFromDCtoNodes() {
	dcsToNodes := []twapi.DCtoNodes{}
	dcsToNodes = append(dcsToNodes, []twapi.DCtoNodes{
		twapi.DCtoNodes{DC: "dc1", Nodes: []string{"host1.justin.tv", "host2.justin.tv"}},
		twapi.DCtoNodes{DC: "dc2", Nodes: []string{"host2.justin.tv", "host3.justin.tv"}},
		twapi.DCtoNodes{DC: "dc3", Nodes: []string{}},
	}...)

	fmt.Println(sortStrings(GetHostnamesFromDCtoNodes(dcsToNodes)))
	// Output:
	// [host1.justin.tv host2.justin.tv host3.justin.tv]
}

func Example_getDatacenterNamesFromDCtoNodes() {
	dcsToNodes := []twapi.DCtoNodes{}
	dcsToNodes = append(dcsToNodes, []twapi.DCtoNodes{
		twapi.DCtoNodes{DC: "dc1", Nodes: []string{"host1.justin.tv", "host2.justin.tv"}},
		twapi.DCtoNodes{DC: "dc2", Nodes: []string{"host2.justin.tv", "host3.justin.tv"}},
		twapi.DCtoNodes{DC: "dc3", Nodes: []string{}},
	}...)
	fmt.Println(GetDatacenterNamesFromDCtoNodes(dcsToNodes))
	// Output:
	// [dc1 dc2 dc3]
}

func Example_mapLookupServiceWithDCs() {
	dcsToNodes, dcsError := mapLookupServiceWithDCs(nil, "", "")
	fmt.Println(dcsToNodes, dcsError)
	// Output:
	// [] []
}

func Example_parseDcName() {
	consulHost, dcName := ParseDcName("wanmaster")
	fmt.Println(consulHost, dcName)
	consulHost, dcName = ParseDcName("myconsul.justin.tv/wanmaster")
	fmt.Println(consulHost, dcName)
	consulHost, dcName = ParseDcName("myconsul.justin.tv:1234/wanmaster")
	fmt.Println(consulHost, dcName)
	// Output:
	// wanmaster
	// myconsul.justin.tv wanmaster
	// myconsul.justin.tv:1234 wanmaster
}

func sortStrings(a []string) []string {
	sort.Strings(a)
	return a
}
