package helpers

import (
	"crypto/tls"
	"net/http"
	"net/url"

	"code.justin.tv/common/chitin"

	"github.com/google/go-github/github"
	"golang.org/x/net/context"
	"golang.org/x/oauth2"
)

func ConfigureGithubClient(baseURL string, token string, ctx context.Context, allowInsecure bool) (*github.Client, error) {
	base, err := url.Parse(baseURL)
	if err != nil {
		return nil, err
	}

	// We need to be able to optionally disable SSL verification for a few test cases
	chitinTransport, err := chitin.RoundTripper(ctx, &http.Transport{
		TLSClientConfig: &tls.Config{
			InsecureSkipVerify: allowInsecure,
		},
	})
	if err != nil {
		return nil, err
	}

	client := github.NewClient(&http.Client{
		Transport: &oauth2.Transport{
			Base:   chitinTransport,
			Source: oauth2.ReuseTokenSource(&oauth2.Token{AccessToken: token}, nil),
		},
	})
	client.BaseURL = base

	return client, nil
}

func GetGitContents(githubClient *github.Client, owner string, name string, path string, ref string) (string, error) {
	file, _, _, err := githubClient.Repositories.GetContents(context.TODO(), owner, name, path, &github.RepositoryContentGetOptions{Ref: ref})
	if err != nil {
		return "", err
	}

	contents, err := file.GetContent()
	if err != nil {
		return "", err
	}

	return contents, nil
}

// GetUserid returns authenticated user info.
func GetGitUser(githubClient *github.Client) (*github.User, error) {
	u, _, err := githubClient.Users.Get(context.TODO(), "")
	if err != nil {
		return nil, err
	}
	return u, nil
}

func GetGitUseridNoError(githubClient *github.Client) string {
	u, err := GetGitUser(githubClient)
	if err != nil || u.Login == nil {
		return ""
	}
	return *u.Login
}
