package helpers

import (
	"github.com/cenkalti/backoff"
	"time"
)

const (
	// The default backoff setting is 15 minutes which is too long.
	// These setting will make it run no longer than 5 seconds which should be enough effort to
	// get over a temporary failure to access remote resources.
	retryIntervalMin = (100 * time.Millisecond)
	retryIntervalMax = (1000 * time.Millisecond)
	maxRetryRuntime  = (15000 * time.Millisecond)
)

// Setting this value will change the retry behavior to linear conventional way. Mainly used in testing
var Retries int = 0

// Retry provides a way to backoff and retry functions when they
// encounter errors. Specifically, this function exists to abstract
// and consolidate the implementation of retrying.
func Retry(f func() error) error {
	if Retries > 0 {
		var err error
		for i := 0; i < Retries; i++ {
			if err = f(); err == nil {
				break
			}
		}
		return err

	}

	b := backoff.NewExponentialBackOff()
	b.InitialInterval = retryIntervalMin
	b.MaxInterval = retryIntervalMax
	b.MaxElapsedTime = maxRetryRuntime
	b.Reset()
	return backoff.Retry(f, b)
}
