package info

import (
	"bytes"
	"encoding/json"
	"expvar"
	"fmt"
	"github.com/gorilla/mux"
	"net/http"
)

// listHandler lists root containers.
func ListHandler(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	b := new(bytes.Buffer)
	b.WriteString("{\"info\":[\"all\"")
	expvar.Do(func(kv expvar.KeyValue) {
		b.WriteString(",")
		b.WriteString(fmt.Sprintf("%q", kv.Key))
	})
	b.WriteString("]}")

	pb := new(bytes.Buffer)
	if err := json.Indent(pb, b.Bytes(), "", "  "); err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		fmt.Fprint(w, err.Error())
		return
	}
	fmt.Fprint(w, pb)
}

// AllHandler shows all the variables in a single json.
func AllHandler(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	b := new(bytes.Buffer)
	b.WriteString("{")
	first := true
	expvar.Do(func(kv expvar.KeyValue) {
		if kv.Key == "memstats" {
			return
		}
		if !first {
			b.WriteString(",")
		}
		b.WriteString(fmt.Sprintf("%q:%s", kv.Key, kv.Value))
		first = false
	})
	b.WriteString("}")

	pb := new(bytes.Buffer)
	if err := json.Indent(pb, b.Bytes(), "", "  "); err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		fmt.Fprint(w, err.Error())
		return
	}
	fmt.Fprint(w, pb)
}

// DetailHandler shows elements under specified root container.
func DetailHandler(w http.ResponseWriter, r *http.Request) {
	root := mux.Vars(r)["root"]

	x := expvar.Get(root)
	if x == nil {
		w.WriteHeader(http.StatusNotFound)
		fmt.Fprintln(w, fmt.Sprintf("Key '%s' not found.", root))
		return
	}

	pb := new(bytes.Buffer)
	if err := json.Indent(pb, []byte(x.String()), "", "  "); err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		fmt.Fprint(w, err.Error())
		return
	}
	fmt.Fprint(w, pb)
}

// DebugHandler shows all the expvar variables without pretty json printing.
// This prints out 100% identical output with the native expvar handler.
func DebugHandler(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	fmt.Fprintf(w, "{\n")
	first := true
	expvar.Do(func(kv expvar.KeyValue) {
		if !first {
			fmt.Fprintf(w, ",\n")
		}
		first = false
		fmt.Fprintf(w, "%q: %s", kv.Key, kv.Value)
	})
	fmt.Fprintf(w, "\n}\n")
}
