// Package info extends `expvar.Map` and provides convenient methods
// to easily manage public variables grouped together.
//
// Basic use of this package is to manage public variables in grouped
// manner under each category.

package info

import (
	"encoding/json"
	"expvar"
	"fmt"
	"reflect"
	"strconv"
	"strings"
)

var (
	sensitiveKeywordsInName  = []string{"token", "secret"}
	sensitiveKeywordsInValue = []string{"password", "passwd"}
)

type Info struct {
	m *expvar.Map
}

// Func implements Var by calling the function
type Func func() interface{}

func (f Func) String() string {
	v, _ := json.Marshal(f())
	return string(v)
}

// PublishRoot declares a named exported variable to the root list.
// The name must be unique.
func PublishRoot(name string, v interface{}) {
	expvar.Publish(name, v.(expvar.Var))
}

// NewCategory creates a variable container with given name.
// This named info container will be exposed via expvar.Publish(),
// so the `name` must be unique.
func NewInfo(name string) *Info {
	i := &Info{
		m: new(expvar.Map),
	}
	i.Init()
	PublishRoot(name, i.m)
	return i
}

// Init removes all keys
func (i *Info) Init() {
	i.m.Init()
}

// SetStruct populates all the fields from a struct and stores them
// as properties. It screens the value of elements which contains
// `filterKeywords` in the name or in the value.
func (i *Info) SetStruct(s interface{}) {
	v := reflect.ValueOf(s).Elem()
	for n := 0; n < v.NumField(); n++ {
		name := v.Type().Field(n).Name
		value := fmt.Sprint(v.Field(n))
		i.SetProperty(name, screenSensitiveData(name, value))
	}
}

func (i *Info) SetProperty(name string, value string) {
	v := new(expvar.String)
	v.Set(value)
	i.m.Set(name, v)
}

// GetProperty returns empty string when there's no such property.
func (i *Info) GetProperty(name string) string {
	v := i.m.Get(name)
	if v == nil {
		return ""
	}
	s := v.String()
	if len(s) > 2 && strings.HasPrefix(s, "\"") && strings.HasSuffix(s, "\"") {
		return s[1 : len(s)-1] // Escape quotes
	}
	return s
}

func (i *Info) SetCounter(name string, value int64) {
	v := new(expvar.Int)
	v.Set(value)
	i.m.Set(name, v)
}

// GetProperty returns 0 when there's no such counter.
func (i *Info) GetCounter(name string) int64 {
	v := i.m.Get(name)
	if v == nil {
		return 0
	}
	num, err := strconv.ParseInt(v.String(), 10, 64)
	if err != nil {
		return 0
	}
	return num
}

func (i *Info) UpdateCounter(name string, delta int64) {
	i.m.Add(name, delta)
}

func (i *Info) IncreaseCounter(name string) {
	i.m.Add(name, 1)
}

func (i *Info) DecreaseCounter(name string) {
	i.m.Add(name, -1)
}

func screenSensitiveData(name string, value string) string {
	for _, keyword := range sensitiveKeywordsInName {
		if caseInsensitiveContains(name, keyword) {
			return "******** (hidden)"
		}
	}
	for _, keyword := range sensitiveKeywordsInValue {
		if caseInsensitiveContains(value, keyword) {
			return "******** (hidden)"
		}
	}
	return value
}

func caseInsensitiveContains(s string, substr string) bool {
	s, substr = strings.ToUpper(s), strings.ToUpper(substr)
	return strings.Contains(s, substr)
}
