package info

import (
	"runtime/debug"
	"sync"
	"testing"
)

func assert(t *testing.T, cond bool) {
	if !(cond) {
		debug.PrintStack()
		t.Fail()
	}
}

func TestStruct(t *testing.T) {
	info := NewInfo("test_struct")

	info.SetStruct(&struct {
		i   int
		i64 int64
		s   string
	}{
		i:   1,
		i64: 2,
		s:   "str",
	})

	assert(t, "1" == info.GetProperty("i"))
	assert(t, "2" == info.GetProperty("i64"))
	assert(t, "str" == info.GetProperty("s"))
}

func TestProperty(t *testing.T) {
	info := NewInfo("test_property")

	info.SetProperty("property", "data")
	assert(t, "data" == info.GetProperty("property"))

	info.SetProperty("property", "data2")
	assert(t, "data2" == info.GetProperty("property"))
}

func TestCounter(t *testing.T) {
	info := NewInfo("test_counter")

	assert(t, 0 == info.GetCounter("counter"))

	info.SetCounter("counter", 10)
	assert(t, 10 == info.GetCounter("counter"))

	info.IncreaseCounter("counter")
	assert(t, 11 == info.GetCounter("counter"))

	info.DecreaseCounter("counter")
	assert(t, 10 == info.GetCounter("counter"))
}

func TestConcurrency(t *testing.T) {
	const Concurrency = 100
	const NumUpdates = 10000

	info := NewInfo("test_concurrency")
	var wgDone, wgTrigger sync.WaitGroup
	wgTrigger.Add(1)
	wgDone.Add(Concurrency)

	for i := 0; i < Concurrency; i++ {
		go func() {
			wgTrigger.Wait() // wait for the trigger
			for n := 0; n < NumUpdates; n++ {
				info.IncreaseCounter("counter")
				info.DecreaseCounter("counter")
				info.UpdateCounter("counter", 1)
			}
			wgDone.Done()
		}()
	}

	wgTrigger.Done() // let all threads run
	wgDone.Wait()    // wait all threads to finish
	assert(t, Concurrency*NumUpdates == info.GetCounter("counter"))
}
