package motd

import (
	"code.justin.tv/dta/skadi/pkg/helpers"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
	"github.com/russross/blackfriday"
	"strings"
	"time"
)

const (
	MOTD_GITHUB_PATH        = "/splash/index.md"
	DEFAULT_BRANCH          = "master"
	MOTD_FETCH_INTERVAL_SEC = 180
)

type Motd struct {
	githubClient  *github.Client
	motdRepoOwner string
	motdRepoName  string

	motdText        string
	motdHtml        string
	motdLastUpdated time.Time
}

func (m *Motd) Init(githubClient *github.Client, motdRepo string) error {
	m.githubClient = githubClient
	s := strings.Split(motdRepo, "/")
	if len(s) != 2 {
		return fmt.Errorf("Invalid repo format - %v", motdRepo)
	}
	m.motdRepoOwner = s[0]
	m.motdRepoName = s[1]

	return nil
}

func (m *Motd) Run() {
	go func() {
		for {
			if err := m.fetchMotd(); err != nil {
				log.Errorln("Failed to fetch MOTD", err)
			}
			time.Sleep(MOTD_FETCH_INTERVAL_SEC * time.Second)
		}
	}()
}

func (m *Motd) fetchMotd() error {
	c, err := helpers.GetGitContents(m.githubClient, m.motdRepoOwner, m.motdRepoName, MOTD_GITHUB_PATH, DEFAULT_BRANCH)
	if err != nil {
		return err
	}
	log.Debugf("MOTD fetched - %v", c)
	m.motdText = c
	m.motdHtml = renderMdToHtml(c)
	m.motdLastUpdated = time.Now()

	return nil
}

func renderMdToHtml(doc string) string {
	return string(blackfriday.MarkdownCommon([]byte(doc)))
}
