package oauth

import (
	"encoding/json"
	"net/http"

	"golang.org/x/oauth2"

	"code.justin.tv/dta/skadi/pkg/config"
)

const (
	OauthScope         = "repo,admin:org_hook"
	GithubAccessCookie = "GithubAccessToken"
	GithubScopeCookie  = "GithubScope"
)

func CheckGithubAuth(f http.HandlerFunc) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		cookie, err := r.Cookie(GithubScopeCookie)
		if err != nil {
			handleAuthorize(w, r)
			return
		}
		if cookie.Value != OauthScope {
			handleAuthorize(w, r)
			return
		}

		_, err = r.Cookie(GithubAccessCookie)
		if err != nil {
			handleAuthorize(w, r)
			return
		}

		f(w, r)
	}
}

func handleAuthorize(w http.ResponseWriter, r *http.Request) {
	http.Redirect(w, r, oauthConfig.AuthCodeURL(""), http.StatusFound)
}

func handleGithubOauth(w http.ResponseWriter, r *http.Request) {
	token, err := oauthConfig.Exchange(oauth2.NoContext, r.FormValue("code"))
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "Unable to Authenticate", err)
		return
	}
	tokenCookie := &http.Cookie{Name: GithubAccessCookie, Value: token.AccessToken}
	scopeCookie := &http.Cookie{Name: GithubScopeCookie, Value: OauthScope}

	http.SetCookie(w, tokenCookie)
	http.SetCookie(w, scopeCookie)
	http.Redirect(w, r, "/", http.StatusFound)
}

func handleConfig(w http.ResponseWriter, r *http.Request) {
	type configStruct struct {
		SkadiHost string `json:"skadi_host"`
	}

	json.NewEncoder(w).Encode(&configStruct{
		SkadiHost: "//",
	})
}
