package repo

import (
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/githubcache"
	"code.justin.tv/dta/skadi/pkg/info"
	"github.com/gorilla/mux"
	consulapi "github.com/hashicorp/consul/api"
	"github.com/pmylund/go-cache"
)

const (
	settingsCacheExpiration      = 30 * time.Second
	settingsCacheCleanupInterval = 5 * time.Minute
)

// CacheStatInfo keeps cache usage statistics.
type CacheStatInfo struct {
	// Since this is statistic data, we don't use locking when updating the counters
	// Stat counters will eventually overflow but it should be ok for enough period
	// between push cycle, so for now it doens't have any protection but should implement
	// overflow handling if needed in the future
	Set    int64
	Get    int64
	GetHit int64
}

var (
	consulClient      *consulapi.Client
	WebhookURL        string
	InsecureGithub    bool
	InstallGitWebhook bool
	settingsCache     *cache.Cache
	cacheStats        *CacheStatInfo
)

func init() {
	settingsCache = cache.New(settingsCacheExpiration, settingsCacheCleanupInterval)
	cacheStats = &CacheStatInfo{}
	info.PublishRoot("pkg_repo_cache", info.Func(getCacheStats))
}

func RegisterHandlers(r *mux.Router, cClient *consulapi.Client) {
	consulClient = cClient

	config.CreateHandler(r, "/v1/repos/{owner}/{name}/settings",
		SettingsHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
	config.CreateHandler(r, "/v1/repos/{owner}/{name}/settings/default",
		createDefaultSettingsHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("POST")
	config.CreateHandler(r, "/v1/repos",
		ListHandler,
		&config.RouteOptions{AddCORS: true},
	).Methods("GET")
}

func SettingsHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)

	vars := mux.Vars(r)
	ref := githubcache.GetDefaultBranch(vars["owner"], vars["name"])
	if r.FormValue("ref") != "" {
		ref = r.FormValue("ref")
	}
	cacheKey := fmt.Sprintf("%s/%s@%s", vars["owner"], vars["name"], ref)
	cacheStats.Get++
	if settings, found := settingsCache.Get(cacheKey); found {
		cacheStats.GetHit++
		json.NewEncoder(w).Encode(settings)
		return
	}
	githubClient, err := config.GithubClient(ctx, true)
	if err != nil {
		config.JSONError(w, http.StatusInternalServerError, "error creating github client", err)
		return
	}

	// read deploy.json from the default branch of the repo
	settings, err := LoadSettings(githubClient, consulClient, vars["owner"], vars["name"], ref)
	if err != nil {
		config.JSONError(w, 500, "error executing load_settings", err)
		return
	}
	settingsCache.Set(cacheKey, settings, cache.DefaultExpiration)
	cacheStats.Set++
	json.NewEncoder(w).Encode(settings)
}

func createDefaultSettingsHandler(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)

	vars := mux.Vars(r)
	repository := &Repository{
		Owner: vars["owner"],
		Name:  vars["name"],
	}

	client, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, 500, "error getting github client", err)
		return
	}

	err = CreateDefaultSettings(client, repository)
	if err != nil {
		config.JSONError(w, 500, "error executing create_default_settings", err)
		return
	}

	w.Write([]byte("null"))
}

func ListHandler(w http.ResponseWriter, r *http.Request) {
	config.JSONError(w, 500, "error executing ListCommand", fmt.Errorf("Not implemented yet."))
}

// Gets called from debug api.
func getCacheStats() interface{} {
	s := &struct {
		Set       int64
		Get       int64
		GetHit    int64
		ItemCount int
		HitRate   float64
	}{
		Set:       cacheStats.Set,
		Get:       cacheStats.Get,
		GetHit:    cacheStats.GetHit,
		ItemCount: settingsCache.ItemCount(),
	}
	if cacheStats.Get > 0 {
		s.HitRate = (float64(cacheStats.GetHit) / float64(cacheStats.Get))
	}

	return s
}
