package repo

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/helpers"
	"code.justin.tv/dta/skadi/pkg/info"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
	consulapi "github.com/hashicorp/consul/api"
)

type Settings struct {
	Deploy   *api.DeployConfig `json:"deploy,omitempty"`
	Warnings *string           `json:"warnings,omitempty"`
}

const (
	InsecureSSLKey = "insecure_ssl"
	ContentTypeKey = "content_type"

	// EnableWebhookUpdate enables updating existing webhooks with current setting.
	// This update is only necessary when we need to change the call-back url.
	EnableWebhookUpdate = false
)

var (
	orgHookExists = make(map[string]bool)
	ghinfo        = info.NewInfo("gitwebhook")

	usersNoHookCheck map[string]bool
)

func CreateDefaultSettings(client *github.Client, repository *Repository) error {
	contents := `
{
	"room": "",
	"job": "",
	"environments": {
		"production": {}
	},
	"consul_services": [],
	"graphite_targets": [],
	"artifact": "tar",
	"restart": {
		"style": "fast"
	},
	"distribution": {
		"style": "fast"
	}
}
`

	commit_msg := `
Add deploy.json from skadi

To learn more about available settings, refer to https://git.xarth.tv/dta/skadi/blob/master/doc/config.md
`
	_, _, err := client.Repositories.CreateFile(context.TODO(), repository.Owner, repository.Name, "deploy.json", &github.RepositoryContentFileOptions{
		Message: &commit_msg,
		Content: []byte(contents),
	})
	if err != nil {
		return err
	}

	return nil
}

// LoadSettings will lookup the settings for a repo. It does this by loading
// deploy.json from the gitrepo and then adding in additional settings like
// dynamic environments from consul.
func LoadSettings(client *github.Client, consul *consulapi.Client, owner, name, ref string) (*Settings, error) {
	warnings := ""
	if InstallGitWebhook {
		warnings, _ = RegisterHooks(client, owner)
	}
	c, err := config.LoadDeployConfig(client, consul, owner, name, ref)
	if err != nil {
		return nil, err
	}

	return &Settings{Deploy: c, Warnings: &warnings}, nil
}

/*
// This method is no more used and commented out because it has hard-coded URL.
// This code is here just for future reference but should not be used before
// taking care of the hard-coded URL.
func DeleteOldHooks(client *github.Client, owner, name string) error {
	hooks, _, err := client.Repositories.ListHooks(owner, name, nil)
	if err != nil {
		log.Printf("Error getting webhooks: %v", err)
		return err
	}

	for _, hook := range hooks {
		if hook.Config["url"] == "http://skadi-0.prod.us-west2.justin.tv:8080/v1/webhook" {
			_, err := client.Repositories.DeleteHook(owner, name, *hook.ID)
			if err != nil {
				log.Printf("Got error deleting old hook for %q: %v", path.Join(owner, name), err)
				return err
			}
		}
	}

	return nil
}
*/

// When skadi is accessed via API with special auth token, sometimes the request to Git doesn't get recognized
// as oauth request so it won't see the installed hook and try to install which could results hook duplication.
// Parameter `users` is comma separated list of git logins.
func AddNoHookCheckUsers(users string) {
	if usersNoHookCheck == nil {
		usersNoHookCheck = make(map[string]bool)
	}
	for _, u := range helpers.SplitStringNoEmpty(users, ",") {
		usersNoHookCheck[u] = true
	}
}

func RegisterHooks(client *github.Client, owner string) (string, error) {
	if orgHookExists[owner] == true {
		log.Debugf("Skipping webhook registration for %v", owner)
		return "", nil
	}

	login := helpers.GetGitUseridNoError(client)
	if usersNoHookCheck != nil && usersNoHookCheck[login] {
		log.Printf("Skipping webhook check for user %s", login)
		return "", nil
	}

	hooks, resp, err := client.Organizations.ListHooks(context.TODO(), owner, nil)
	if err != nil {
		w := ""
		if resp != nil && resp.StatusCode == http.StatusNotFound {
			// 404 return code indicates that the user doesn't have permission
			// to the organization. Only users listed ad admin can set up hooks
			// others will be using existing hooks. So this is not an error but
			// intended behavior.
			// https://developer.github.com/v3/troubleshooting/#why-am-i-getting-a-404-error-on-a-repository-that-exists
			//
			// It is possible we get real 404 due to misconfiguration
			// which is a bigger problem but in such a case, it will be
			// caught from another logic.
			log.Debug("Could not list webhooks for ", owner, " - ", err)
		} else {
			// All other errors should be reported to users.
			w = fmt.Sprintf("Error listing webhooks for %q - %v", owner, err)
			log.Println(w)
		}
		return w, err
	}
	log.Printf("Installed webhooks for %v: %+v", owner, hooks)

	// If it reaches here, the user has admin permission in this org.
	// Webhook needs to be created only once after org creation and most of
	// time it doesn't need to be updated unless we need to change the callback
	// url. So here we allow to create or update org-webhook only once per org
	// since after start up.
	//
	// This will cut down unnecessary Git api calls also reduces the chance of creating
	// duplicated webhooks with old oauth ID. The chance is slim unless the wrong
	// user happens to be the 1st person to access the org.
	//
	// Also this change requires you to bounce Skadi if someone deletes webhook manually
	// from GitHub but it should be rare.
	for _, hook := range hooks {
		// we will be able to see only the webhooks we've installed,
		// so precise comparison is not necessary but just check the name.
		if hook.ID != nil {
			log.Printf("Webhook exists for %s: %+v", owner, hook)
			ghinfo.SetProperty(fmt.Sprintf("%s-exist", owner), time.Now().String())

			if EnableWebhookUpdate {
				w, err := UpdateHook(client, owner, hook)
				if err == nil {
					orgHookExists[owner] = true

					ghinfo.IncreaseCounter(fmt.Sprintf("%s-update", owner))
					ghinfo.SetProperty(fmt.Sprintf("%s-update-%s-%d", owner, login, time.Now().Unix()), time.Now().String())
				}
				log.Printf("Updating webhooks for %s: warnings: %v error: %v", owner, w, err)
				return w, err
			}
			orgHookExists[owner] = true
			return "", nil
		}
	}

	w, err := CreateHook(client, owner)
	if err == nil {
		orgHookExists[owner] = true
		ghinfo.IncreaseCounter(fmt.Sprintf("%s-create", owner))
		ghinfo.SetProperty(fmt.Sprintf("%s-create-%s-%d", owner, login, time.Now().Unix()), time.Now().String())
	}
	log.Printf("Creating webhooks for %s: warnings: %v error: %v", owner, w, err)
	return w, err
}

func CreateHook(client *github.Client, owner string) (string, error) {
	warnings := ""
	hookConfig := make(map[string]interface{})
	hookConfig["url"] = WebhookURL
	hookConfig[ContentTypeKey] = "json"
	hookConfig[InsecureSSLKey] = helpers.Btoi(InsecureGithub)
	new_hook := github.Hook{Events: []string{"*"}, Config: hookConfig}
	log.Printf("Creating webhook for %v: %+v", owner, new_hook)
	_, _, err := client.Organizations.CreateHook(context.TODO(), owner, &new_hook)
	if err != nil {
		warnings = fmt.Sprintf("Error adding webhook: %v", err)
		log.Printf(warnings)
		return warnings, err
	}
	return warnings, nil
}

func UpdateHook(client *github.Client, owner string, hook *github.Hook) (string, error) {
	warnings := ""
	changes := false

	url, found := hook.Config["url"]
	if u, ok := url.(string); !found || !ok || u != WebhookURL {
		changes = true
		hook.Config["url"] = WebhookURL
	}

	insecureSSL, found := hook.Config[InsecureSSLKey]
	if is, ok := insecureSSL.(int); !found || !ok || is != helpers.Btoi(InsecureGithub) {
		changes = true
		hook.Config[InsecureSSLKey] = helpers.Btoi(InsecureGithub)
	}

	if changes {
		hook.Config[ContentTypeKey] = "json"
		_, _, err := client.Organizations.EditHook(context.TODO(), owner, *hook.ID, hook)
		if err != nil {
			warnings = fmt.Sprintf("Error updating webhook: %v", err)
			log.Printf(warnings)
			return warnings, err
		}
	}

	return warnings, nil
}
