package rtevent_receiver

import (
	"encoding/json"
	"fmt"
	"io"
	"os"
	"strings"
	"sync"

	"code.justin.tv/dta/rtevent"
	log "github.com/Sirupsen/logrus"
	"github.com/jmoiron/sqlx"
)

const (
	maxPullNum  = 50
	maxPullWait = 5

	COURIER_PHASE_INSTALL_LOCAL  = "install-local"
	COURIER_PHASE_RESTART_LOCAL  = "restart-local"
	COURIER_PHASE_INSTALL_REMOTE = "install-remote"
	COURIER_PHASE_RESTART_REMOTE = "restart-remote"

	STATUS_SUCCESS = "success"
	STATUS_FAILURE = "failure"
	STATUS_UNKNOWN = "unknown"
)

type RteventReceiverGlobal struct {
	once sync.Once
}

var (
	db *DB
	rr RteventReceiverGlobal
)

func LaunchRteventReceiverThread(dbInstance *sqlx.DB, logPath string) error {
	db = NewDB(dbInstance)

	reterr := fmt.Errorf("Already running")
	rr.once.Do(func() {
		log.Println("Starting rtevent receiver")
		var out io.Writer
		if logPath != "" {
			if o, err := os.OpenFile(logPath, os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644); err != nil {
				log.Errorf("Could not open rtevent log at %v", logPath)
			} else {
				out = io.Writer(o)
			}
		} else {
			out = nil
		}

		client, err := rtevent.NewClient(nil, out)
		if err != nil {
			reterr = err
			return
		}

		go func(client *rtevent.Client) {
			for {
				events, err := client.PullEvent(maxPullNum, maxPullWait)
				if err != nil {
					log.Errorf("Failed to pull events - %v", err)
				}
				if err = storeEvent(events); err != nil {
					log.Errorf("Failed to store an event - %v", err)
				}
			}
		}(client)
		reterr = nil
	})
	return reterr
}

func storeEvent(events []string) error {
	for _, event := range events {
		byteEvent := []byte(event)
		e := rtevent.EventHeader{}
		if err := json.Unmarshal(byteEvent, &e); err != nil {
			log.Warnf("Failed to unmarshal - %v", event)
			continue
		}

		log.Debugf("event received - %+v", e)

		if e.Version != rtevent.EventMessageVersion {
			log.Warnf("Invalid header with wrong version number - %+v", e)
			continue
		}

		switch e.Type {
		case rtevent.EventTypeDeploy:
			deployEvent := rtevent.DeployEvent{}
			if err := json.Unmarshal(byteEvent, &deployEvent); err != nil {
				log.Warnf("Failed to unmarshal DeployEvent - %v", event)
				continue
			}

			storeDeployEvent(&deployEvent)
		}
	}
	return nil
}

func storeDeployEvent(event *rtevent.DeployEvent) error {
	log.Debugf("DeployEvent=> %+v\n", event)

	if event.Deployer == "skadi" && event.DeployID > 0 {
		// skadi event, sent from master-courier
		if err := db.UpdateSkadiDeployStatus(event); err != nil {
			log.Errorf("Failed to update db - %v", err)
			return err
		}
	} else {
		// courier event, sent from local-courier
		if !((event.Phase == "" || event.Phase == COURIER_PHASE_INSTALL_LOCAL) && event.Success == true) {
			return nil
		}
		if err := db.UpdateHostVersion(event); err != nil {
			log.Errorf("Failed to update db - %v", err)
			return err
		}
	}

	return nil
}

func parseAppStr(app string) (string, string, error) {
	s := strings.Split(app, "/")
	if len(s) != 2 {
		return "", "", fmt.Errorf("Invalid format - %v", app)
	}
	if len(s[0]) == 0 || len(s[1]) == 0 {
		return "", "", fmt.Errorf("Invalid format - %v", app)
	}
	return s[0], s[1], nil
}
