package subscriptions

import (
	"database/sql"
	"github.com/jmoiron/sqlx"
	"log"
)

type DB struct {
	*sqlx.DB
}

/* helper method for row retrieval */
func processRows(rows *sql.Rows) (*[]Subscription, error) {
	subs := []Subscription{}
	for rows.Next() {
		var s Subscription
		err := rows.Scan(&s.Org, &s.Email, &s.Success, &s.Pending, &s.Failure, &s.Environment)
		if err != nil {
			log.Printf("Got error on generating subscription object: %+v", err)
			return nil, err
		}
		subs = append(subs, s)
	}
	return &subs, nil
}

func (db *DB) InsertSubscription(s *Subscription) error {
	log.Printf("Inserting a subscription: %+v", s)
	_, err := db.Exec("INSERT INTO subscriptions (org, email, success, pending, failure, environment) VALUES ($1, $2, $3, $4, $5, $6)",
		s.Org, s.Email, s.Success, s.Pending, s.Failure, s.Environment)
	if err != nil {
		log.Printf("Got error on subscription insert: %+v", err)
		return err
	}
	return nil
}

func (db *DB) UpdateSubscription(s *Subscription) error {
	log.Printf("Updating a subscription: %+v", s)
	_, err := db.Exec("UPDATE subscriptions SET success=$1, pending=$2, failure=$3 WHERE org=$4 and email=$5 and environment=$6", s.Success, s.Pending, s.Failure, s.Org, s.Email, s.Environment)
	if err != nil {
		log.Printf("Got error on subscription update: %+v", err)
		return err
	}
	return nil
}

func (db *DB) GetAllSubscriptions(org *string) (*[]Subscription, error) {
	rows, err := db.Query("SELECT org, email, success, pending, failure, environment FROM subscriptions WHERE org=$1", org)
	if err != nil {
		log.Printf("Got error on selecting subscriptions: %+v", err)
		return nil, err
	} else {
		defer rows.Close()
		return processRows(rows)
	}
}

/* Returns rows subscribed to this environment*/
func (db *DB) GetSubscriptionsForEnv(org *string, env *string) (*[]Subscription, error) {
	rows, err := db.Query("SELECT org, email, success, pending, failure, environment FROM subscriptions WHERE org=$1 and (environment=$2 or environment='')", org, env)
	if err != nil {
		log.Printf("Got error on selecting subscriptions: %+v", err)
		return nil, err
	} else {
		defer rows.Close()
		return processRows(rows)
	}
}

func (db *DB) DeleteSubscription(s *Subscription) error {
	log.Printf("Deleting a subscription: %+v", s)
	_, err := db.Exec("DELETE FROM subscriptions WHERE org=$1 and email=$2 and environment=$3", s.Org, s.Email, s.Environment)
	if err != nil {
		log.Printf("Got error on subscription deletion: %+v", err)
		return err
	}
	return nil
}
