package subscriptions

import (
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"net/http"

	"code.justin.tv/dta/skadi/pkg/config"
	"github.com/gorilla/mux"
	"github.com/jmoiron/sqlx"
)

type Subscription struct {
	Org         string `json:"org" db:"org"`
	Email       string `json:"email" db:"email"`
	Success     bool   `json:"success" db:"success"`
	Pending     bool   `json:"pending" db:"pending"`
	Failure     bool   `json:"failure" db:"failure"`
	Environment string `json:"environment" db:"environment"`
}

var (
	db *DB
)

func RegisterHandlers(r *mux.Router, dbInstance *sqlx.DB) {
	db = &DB{dbInstance}

	r.HandleFunc("/v1/subscriptions/", config.AddCORSHeaders(getSubscriptionsHandler)).Methods("GET")
	r.HandleFunc("/v1/subscriptions/create", config.AddCORSHeaders(createSubscriptionsHandler)).Methods("POST")
	r.HandleFunc("/v1/subscriptions/update", config.AddCORSHeaders(updateSubscriptionsHandler)).Methods("POST")
	r.HandleFunc("/v1/subscriptions/delete", config.AddCORSHeaders(deleteSubscriptionsHandler)).Methods("POST")
}

func GetAllSubscriptions(org *string) (*[]Subscription, error) {
	subs, err := db.GetAllSubscriptions(org)
	return subs, err
}

func GetSubscriptionsForEnv(org *string, env *string) (*[]Subscription, error) {
	subs, err := db.GetSubscriptionsForEnv(org, env)
	return subs, err
}

func getSubscriptionsHandler(w http.ResponseWriter, r *http.Request) {
	log.Printf("Getting subscription: %v", &r.Body)
	org := r.FormValue("org")
	if org == "" {
		err := errors.New("'org' parameter required")
		config.JSONError(w, 400, err.Error(), err)
		return
	}
	subs, err := db.GetAllSubscriptions(&org)
	if err != nil {
		config.JSONError(w, 500, fmt.Sprintf("Error retrieving org subscriptions for %v. Error: %v", org, err), err)
		return
	}
	subsJson, err := json.Marshal(subs)
	if err != nil {
		config.JSONError(w, 500, fmt.Sprintf("Error marshalling subscriptions %v. Error: %v", org, err), err)
		return
	}
	w.Write(subsJson)
}

func createSubscriptionsHandler(w http.ResponseWriter, r *http.Request) {
	log.Printf("Creating subscription: %v", &r.Body)
	var s Subscription
	err := json.NewDecoder(r.Body).Decode(&s)
	if err != nil {
		config.JSONError(w, 500, "createSubscriptionsHandler unable to decode JSON", err)
		return
	}
	if s.Org == "" {
		err := errors.New("New subscription has no org")
		config.JSONError(w, 400, err.Error(), err)
		return
	} else if s.Email == "" {
		err := errors.New("New subscription has no email")
		config.JSONError(w, 400, err.Error(), err)
		return
	}
	subs, err := db.GetAllSubscriptions(&s.Org)
	if err != nil {
		config.JSONError(w, 500, fmt.Sprintf("Error retrieving org subscriptions for %v. Error: %v", s.Org, err), err)
		return
	}
	for _, sub := range *subs {
		//NOTE: This allows us to have null as environment as well as any number of rows with an environment set.
		// In this case, the set environment rows will act as overrides
		if (sub.Email == s.Email) && (sub.Environment == s.Environment) {
			config.JSONError(w, 400, "Email already exists for this environment", err)
			return
		}
	}
	err = db.InsertSubscription(&s)
	if err != nil {
		config.JSONError(w, 500, "Error inserting a subscription", err)
		return
	}
	w.Write([]byte("ok"))
}

func updateSubscriptionsHandler(w http.ResponseWriter, r *http.Request) {
	log.Printf("Updating subscription: %v", &r.Body)
	decoder := json.NewDecoder(r.Body)
	var s Subscription
	err := decoder.Decode(&s)
	if err != nil {
		config.JSONError(w, 500, "updateSubscriptionsHandler unable to decode JSON", err)
		return
	}
	if s.Org == "" {
		err := errors.New("Subscription to edit has no org")
		config.JSONError(w, 400, err.Error(), err)
		return
	} else if s.Email == "" {
		err := errors.New("Subscription to edit has no email")
		config.JSONError(w, 400, err.Error(), err)
		return
	}
	err = db.UpdateSubscription(&s)
	if err != nil {
		config.JSONError(w, 500, "Error updating a subscription", err)
		return
	}
	w.Write([]byte("ok"))
}

func deleteSubscriptionsHandler(w http.ResponseWriter, r *http.Request) {
	log.Printf("Deleting subscription: %v", &r.Body)
	decoder := json.NewDecoder(r.Body)
	var s Subscription
	err := decoder.Decode(&s)
	if err != nil {
		config.JSONError(w, 500, "deleteSubscriptionsHandler unable to decode JSON", err)
		return
	}
	if s.Org == "" {
		err := errors.New("Must specify org")
		config.JSONError(w, 400, err.Error(), err)
		return
	} else if s.Email == "" {
		err := errors.New("Must specify email")
		config.JSONError(w, 400, err.Error(), err)
		return
	}
	err = db.DeleteSubscription(&s)
	if err != nil {
		config.JSONError(w, 500, "Error deleting a subscription", err)
		return
	}
	w.Write([]byte("ok"))
}
