package user

import (
	"context"
	"encoding/json"
	"net/http"
	"strconv"
	"sync"
	"time"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/dta/skadi/pkg/config"
	"github.com/google/go-github/github"
)

type RepoError struct {
	Message string
	Error   error
}

type SearchResult struct {
	SearchResult *github.CodeSearchResult
	Response     *github.Response
	Error        error
}

type RepoResult struct {
	Repositories []*github.Repository
	Response     *github.Response
	Error        error
}

// Process a group of search api calls. Return the last processed page.
func processSearch(startPage int, lastPage int, perPage int, wg *sync.WaitGroup, client *github.Client, results chan SearchResult) {
	for currentPage := startPage; currentPage <= lastPage; currentPage++ {
		wg.Add(1)
		go func(pageNum int) {
			defer wg.Done()
			searchResult, response, err := client.Search.Code(
				context.TODO(),
				"path:/ filename:deploy extension:json",
				&github.SearchOptions{
					Sort:  "indexed",
					Order: "desc",
					ListOptions: github.ListOptions{
						PerPage: perPage,
						Page:    pageNum,
					}},
			)
			results <- SearchResult{SearchResult: searchResult, Response: response, Error: err}
		}(currentPage)
	}
}

// Process a group of repository api calls. Return the last processed page.
func processRepo(startPage int, lastPage int, perPage int, wg *sync.WaitGroup, client *github.Client, results chan RepoResult) {
	for currentPage := startPage; currentPage <= lastPage; currentPage++ {
		wg.Add(1)
		go func(pageNum int) {
			defer wg.Done()
			repositories, response, err := client.Repositories.List(
				context.TODO(),
				"",
				&github.RepositoryListOptions{
					ListOptions: github.ListOptions{
						Page:    pageNum,
						PerPage: perPage}})
			results <- RepoResult{Repositories: repositories, Response: response, Error: err}
		}(currentPage)
	}
}

func getDeployableRepos(perPage int, client *github.Client) ([]RepoError, []*github.Repository) {
	var wg sync.WaitGroup
	start := time.Now()

	errorList := []RepoError{}
	parallelRequests := 3
	channelSize := perPage * parallelRequests
	deployableRepos := []*github.Repository{}
	lastPageSearch := parallelRequests
	lastPageRepo := parallelRequests
	deployableReposMap := map[string]*github.Repository{}
	deployableRepoNames := map[string]bool{}
	// Parallelize the API requests to github, under a `parallelRequests`
	// limit. The original implementation of this function made the
	// right API calls, but they depended on each other and had a serial
	// algorithm. This new version uses a separate endpoint that isn't
	// as precise, but gives us all the data we needed anyway, so we just
	// quickly gather everything we need as fast as possible, and then
	// do all the matching in code.
	for startPage := 1; startPage <= lastPageSearch || startPage <= lastPageRepo; startPage += parallelRequests {
		searchResults := make(chan SearchResult, channelSize)
		repoResults := make(chan RepoResult, channelSize)
		lastPage := startPage + parallelRequests - 1
		if startPage <= lastPageSearch {
			processSearch(startPage, lastPage, perPage, &wg, client, searchResults)
		}
		if startPage <= lastPageRepo {
			processRepo(startPage, lastPage, perPage, &wg, client, repoResults)
		}
		wg.Wait()
		close(searchResults)
		close(repoResults)
		for groupResult := range searchResults {
			for _, result := range groupResult.SearchResult.CodeResults {
				deployableRepoNames[*result.Repository.FullName] = true
			}
			if groupResult.Response.LastPage > lastPageSearch {
				// Update last page of search pages to adjust number of API hits
				lastPageSearch = groupResult.Response.LastPage
			}
			if groupResult.Error != nil {
				errorList = append(errorList, RepoError{Message: "error fetching deployable repo", Error: groupResult.Error})
			}
		}
		for groupResult := range repoResults {
			for _, result := range groupResult.Repositories {
				deployableReposMap[*result.FullName] = result
			}
			if groupResult.Response.LastPage > lastPageRepo {
				// Update last page of repository pages to adjust number of API hits
				lastPageRepo = groupResult.Response.LastPage
			}
			if groupResult.Error != nil {
				errorList = append(errorList, RepoError{Message: "error fetching repo", Error: groupResult.Error})
			}
		}
	}
	for repoName, repo := range deployableReposMap {
		if deployableRepoNames[repoName] {
			deployableRepos = append(deployableRepos, repo)
		}
	}
	log.Warn("Parallelized ListReposForUser took ", time.Since(start), " to find ", len(deployableRepos), " matches with ", perPage, " per page.")
	return errorList, deployableRepos
}

func ListReposForUser(w http.ResponseWriter, r *http.Request) {
	ctx := config.GetContext(w, r)
	client, err := config.GithubClient(ctx, false)
	if err != nil {
		config.JSONError(w, 500, "error creating github client", err)
		return
	}
	perPage, err := strconv.Atoi(r.FormValue("per_page"))
	if err != nil {
		log.Printf("error parsing per_page: %v", err)
		config.JSONError(w, 500, "error parsing per_page", err)
		return
	}

	errorList, deployableRepos := getDeployableRepos(perPage, client)
	// Search can report errors but deployableRepos will still have valid data most of cases,
	// so log them and move on.
	if len(errorList) > 0 {
		for _, repoError := range errorList {
			log.Warnln(repoError.Error.Error())
		}
	}
	json.NewEncoder(w).Encode(deployableRepos)
}
